Write-Host "Configuring boot logo"

$toolsPath = "$env:SystemDrive\Tools\"

# check if enabled
$path = (Get-Partition | ? IsSystem).AccessPaths[0]
if (-not (Test-Path -LiteralPath $path\EFI\HackBGRT)) {
    Write-Host "Custom bootlogo not enabled, enabling..."
    Start-Process -FilePath "$toolsPath\HackBGRT\setup.exe" -ArgumentList "batch disable install enable-bcdedit" -Verb runas -Wait 2>&1
    # will now reboot after user confirms
    Write-Host "System needs to reboot to enable custom bootlogo. Run 'Provision Device' again after reboot."  -ForegroundColor Green
    $confirmation = Read-Host "Press Enter to continue or CTRL+C to cancel"
    Write-Host "Rebooting to enable custom bootlogo..."
    Restart-Computer
} else {
    Write-Host "Custom bootlogo already enabled"
}
# SIG # Begin signature block
# MII+HgYJKoZIhvcNAQcCoII+DzCCPgsCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAjA4FO0nBYzb9m
# 2IsUzZsW3qp1gb4eSAZ+uX9iQm1OjaCCIuAwggXMMIIDtKADAgECAhBUmNLR1FsZ
# lUgTecgRwIeZMA0GCSqGSIb3DQEBDAUAMHcxCzAJBgNVBAYTAlVTMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jvc29mdCBJZGVu
# dGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAy
# MDAeFw0yMDA0MTYxODM2MTZaFw00NTA0MTYxODQ0NDBaMHcxCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jv
# c29mdCBJZGVudGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRo
# b3JpdHkgMjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALORKgeD
# Bmf9np3gx8C3pOZCBH8Ppttf+9Va10Wg+3cL8IDzpm1aTXlT2KCGhFdFIMeiVPvH
# or+Kx24186IVxC9O40qFlkkN/76Z2BT2vCcH7kKbK/ULkgbk/WkTZaiRcvKYhOuD
# PQ7k13ESSCHLDe32R0m3m/nJxxe2hE//uKya13NnSYXjhr03QNAlhtTetcJtYmrV
# qXi8LW9J+eVsFBT9FMfTZRY33stuvF4pjf1imxUs1gXmuYkyM6Nix9fWUmcIxC70
# ViueC4fM7Ke0pqrrBc0ZV6U6CwQnHJFnni1iLS8evtrAIMsEGcoz+4m+mOJyoHI1
# vnnhnINv5G0Xb5DzPQCGdTiO0OBJmrvb0/gwytVXiGhNctO/bX9x2P29Da6SZEi3
# W295JrXNm5UhhNHvDzI9e1eM80UHTHzgXhgONXaLbZ7LNnSrBfjgc10yVpRnlyUK
# xjU9lJfnwUSLgP3B+PR0GeUw9gb7IVc+BhyLaxWGJ0l7gpPKWeh1R+g/OPTHU3mg
# trTiXFHvvV84wRPmeAyVWi7FQFkozA8kwOy6CXcjmTimthzax7ogttc32H83rwjj
# O3HbbnMbfZlysOSGM1l0tRYAe1BtxoYT2v3EOYI9JACaYNq6lMAFUSw0rFCZE4e7
# swWAsk0wAly4JoNdtGNz764jlU9gKL431VulAgMBAAGjVDBSMA4GA1UdDwEB/wQE
# AwIBhjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTIftJqhSobyhmYBAcnz1AQ
# T2ioojAQBgkrBgEEAYI3FQEEAwIBADANBgkqhkiG9w0BAQwFAAOCAgEAr2rd5hnn
# LZRDGU7L6VCVZKUDkQKL4jaAOxWiUsIWGbZqWl10QzD0m/9gdAmxIR6QFm3FJI9c
# Zohj9E/MffISTEAQiwGf2qnIrvKVG8+dBetJPnSgaFvlVixlHIJ+U9pW2UYXeZJF
# xBA2CFIpF8svpvJ+1Gkkih6PsHMNzBxKq7Kq7aeRYwFkIqgyuH4yKLNncy2RtNwx
# AQv3Rwqm8ddK7VZgxCwIo3tAsLx0J1KH1r6I3TeKiW5niB31yV2g/rarOoDXGpc8
# FzYiQR6sTdWD5jw4vU8w6VSp07YEwzJ2YbuwGMUrGLPAgNW3lbBeUU0i/OxYqujY
# lLSlLu2S3ucYfCFX3VVj979tzR/SpncocMfiWzpbCNJbTsgAlrPhgzavhgplXHT2
# 6ux6anSg8Evu75SjrFDyh+3XOjCDyft9V77l4/hByuVkrrOj7FjshZrM77nq81YY
# uVxzmq/FdxeDWds3GhhyVKVB0rYjdaNDmuV3fJZ5t0GNv+zcgKCf0Xd1WF81E+Al
# GmcLfc4l+gcK5GEh2NQc5QfGNpn0ltDGFf5Ozdeui53bFv0ExpK91IjmqaOqu/dk
# ODtfzAzQNb50GQOmxapMomE2gj4d8yu8l13bS3g7LfU772Aj6PXsCyM2la+YZr9T
# 03u4aUoqlmZpxJTG9F9urJh4iIAGXKKy7aIwggcEMIIE7KADAgECAhMzAAYaeTMd
# 9e5ebamiAAAABhp5MA0GCSqGSIb3DQEBDAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMTIk1pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIwHhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0
# MjAzMzAyWjCBgDELMAkGA1UEBhMCR0IxDzANBgNVBAgTBkRvcnNldDEOMAwGA1UE
# BxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0
# ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBEYXRhIE1hbmFnZW1lbnQgTHRkMIIBojAN
# BgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAmcc32FHGLV4FiKzse5eN9PjhviRa
# CblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK6e9cz2ayR0wdaIJSElN50pHTPrWMDQSL
# P7wkC1LovyoEhNzVb40CoJhCTXQTkHqnACMD9NqeDdyaF4r0Nb0u716+W67pv/CA
# CE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZVrqE61MYKeWlQWErF2VSZJaA8X9pJuvH
# 1WxVB+Mr7LIehjb9cxmUfelY1LkKimcVkdpSlZsBCZVANliqCugFrqMLJ1jQRmpd
# 8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp/hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8
# vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgXTHt5262ALFRyo3/NShidyL3k8a4hSFwD
# tWFHtXJv7HKGxbCCZdR8TGMFupw4IeWSBdApLGInTzSMhDxAyTK26McpCGRe3c1z
# BOw753ZIhx5k0LhWztmdCXWZIh9h1UBySzdDAgMBAAGjggIaMIICFjAMBgNVHRMB
# Af8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9BgNVHSUENjA0BgorBgEEAYI3YQEABggr
# BgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQU
# uRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYDVR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQW
# hCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZWaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSUQlMjBWZXJpZmllZCUyMENTJTIwQU9D
# JTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUFBwEBBIGYMIGVMGQGCCsGAQUFBzAChlho
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUy
# MElEJTIwVmVyaWZpZWQlMjBDUyUyMEFPQyUyMENBJTIwMDIuY3J0MC0GCCsGAQUF
# BzABhiFodHRwOi8vb25lb2NzcC5taWNyb3NvZnQuY29tL29jc3AwZgYDVR0gBF8w
# XTBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMAgGBmeBDAEEATAN
# BgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKMPRNLuXymFInLM6+tzuw0G5i8qcvMna7j
# ltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYcn19Kvk5bdbRQW50EmDEdH5ybIuE1v5VP
# OjGAg9xvXFk1RdISpST9ghRE+JpdI2gx1uKGUGFCjll6nU/WuFpskg2jV6HQVg7t
# cdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBGDrfJJwLty0ageLObO88+fJdES7ZYsmC1
# Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9yQ+UTijQVYml/vi4oOLROYVk9Auf31WB5
# qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U++cDFVsMH6hvGXrByof7kuE6BQbYk2T0
# 8MaNus4xleApIAYyWr8Prtwe71+T2S+OM/NE+T0JwATBrSwUjBftUzM9nQ3iHGJW
# gT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcbQmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1
# Xjau+kf/EJeYzrhw1H5++1xVbSL1a+BudVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5o
# ERoxq92UpXtQYMo8AOC0TqmdFsqrAmmqi+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJ
# wyGaI2eSqCeROzqTE/BUP6ICU26FD03Mr1QnmMVAN3ligIAAAHHTMJmNaLHVuhVU
# DCAwggcEMIIE7KADAgECAhMzAAYaeTMd9e5ebamiAAAABhp5MA0GCSqGSIb3DQEB
# DAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xKzApBgNVBAMTIk1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIw
# HhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0MjAzMzAyWjCBgDELMAkGA1UEBhMCR0Ix
# DzANBgNVBAgTBkRvcnNldDEOMAwGA1UEBxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0
# aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBE
# YXRhIE1hbmFnZW1lbnQgTHRkMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC
# AYEAmcc32FHGLV4FiKzse5eN9PjhviRaCblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK
# 6e9cz2ayR0wdaIJSElN50pHTPrWMDQSLP7wkC1LovyoEhNzVb40CoJhCTXQTkHqn
# ACMD9NqeDdyaF4r0Nb0u716+W67pv/CACE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZ
# VrqE61MYKeWlQWErF2VSZJaA8X9pJuvH1WxVB+Mr7LIehjb9cxmUfelY1LkKimcV
# kdpSlZsBCZVANliqCugFrqMLJ1jQRmpd8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp
# /hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgX
# THt5262ALFRyo3/NShidyL3k8a4hSFwDtWFHtXJv7HKGxbCCZdR8TGMFupw4IeWS
# BdApLGInTzSMhDxAyTK26McpCGRe3c1zBOw753ZIhx5k0LhWztmdCXWZIh9h1UBy
# SzdDAgMBAAGjggIaMIICFjAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9
# BgNVHSUENjA0BgorBgEEAYI3YQEABggrBgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw
# 4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQUuRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYD
# VR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQWhCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZW
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIw
# SUQlMjBWZXJpZmllZCUyMENTJTIwQU9DJTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUF
# BwEBBIGYMIGVMGQGCCsGAQUFBzAChlhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMElEJTIwVmVyaWZpZWQlMjBDUyUyMEFP
# QyUyMENBJTIwMDIuY3J0MC0GCCsGAQUFBzABhiFodHRwOi8vb25lb2NzcC5taWNy
# b3NvZnQuY29tL29jc3AwZgYDVR0gBF8wXTBRBgwrBgEEAYI3TIN9AQEwQTA/Bggr
# BgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9Eb2NzL1Jl
# cG9zaXRvcnkuaHRtMAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKM
# PRNLuXymFInLM6+tzuw0G5i8qcvMna7jltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYc
# n19Kvk5bdbRQW50EmDEdH5ybIuE1v5VPOjGAg9xvXFk1RdISpST9ghRE+JpdI2gx
# 1uKGUGFCjll6nU/WuFpskg2jV6HQVg7tcdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBG
# DrfJJwLty0ageLObO88+fJdES7ZYsmC1Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9y
# Q+UTijQVYml/vi4oOLROYVk9Auf31WB5qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U
# ++cDFVsMH6hvGXrByof7kuE6BQbYk2T08MaNus4xleApIAYyWr8Prtwe71+T2S+O
# M/NE+T0JwATBrSwUjBftUzM9nQ3iHGJWgT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcb
# QmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1Xjau+kf/EJeYzrhw1H5++1xVbSL1a+Bu
# dVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5oERoxq92UpXtQYMo8AOC0TqmdFsqrAmmq
# i+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJwyGaI2eSqCeROzqTE/BUP6ICU26FD03M
# r1QnmMVAN3ligIAAAHHTMJmNaLHVuhVUDCAwggdaMIIFQqADAgECAhMzAAAABJZQ
# S9Lb7suIAAAAAAAEMA0GCSqGSIb3DQEBDAUAMGMxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xNDAyBgNVBAMTK01pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDb2RlIFNpZ25pbmcgUENBIDIwMjEwHhcNMjEwNDEzMTczMTUy
# WhcNMjYwNDEzMTczMTUyWjBaMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSswKQYDVQQDEyJNaWNyb3NvZnQgSUQgVmVyaWZpZWQg
# Q1MgQU9DIENBIDAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA4c6g
# 6DOiY6bAOwCPbBlQF2tjo3ckUZuab5ZorMnRp4rOmwZDiTbIpzFkZ/k8k4ivBJV1
# w5/b/oykI+eXAqaaxMdyAO0ModnEW7InfQ+rTkykEzHxRbCNg6KDsTnYc/YdL7II
# iJli8k51upaHLL7CYm9YNc0SFYvlaFj2O0HjO9y/NRmcWNjamZOlRjxW2cWgUsUd
# azSHgRCek87V2bM/17b+o8WXUW91IpggRasmiZ65WEFHXKbyhm2LbhBK6ZWmQoFe
# E+GWrKWCGK/q/4RiTaMNhHXWvWv+//I58UtOxVi3DaK1fQ6YLyIIGHzD4CmtcrGi
# vxupq/crrHunGNB7//Qmul2ZP9HcOmY/aptgUnwT+20g/A37iDfuuVw6yS2Lo0/k
# p/jb+J8vE4FMqIiwxGByL482PMVBC3qd/NbFQa8Mmj6ensU+HEqv9ar+AbcKwumb
# ZqJJKmQrGaSNdWfk2NodgcWOmq7jyhbxwZOjnLj0/bwnsUNcNAe09v+qiozyQQes
# 8A3UXPcRQb8G+c0yaO2ICifWTK7ySuyUJ88k1mtN22CNftbjitiAeafoZ9Vmhn5R
# fb+S/K5arVvTcLukt5PdTDQxl557EIE6A+6XFBpdsjOzkLzdEh7ELk8PVPMjQfPC
# gKtJ84c17fd2C9+pxF1lEQUFXY/YtCL+Nms9cWUCAwEAAaOCAg4wggIKMA4GA1Ud
# DwEB/wQEAwIBhjAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUJEWZoXeQKnzD
# yoOwbmQWhCr4LGcwVAYDVR0gBE0wSzBJBgRVHSAAMEEwPwYIKwYBBQUHAgEWM2h0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0
# bTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTASBgNVHRMBAf8ECDAGAQH/AgEA
# MB8GA1UdIwQYMBaAFNlBKbAPD2Ns72nX9c0pnqRIajDmMHAGA1UdHwRpMGcwZaBj
# oGGGX2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29m
# dCUyMElEJTIwVmVyaWZpZWQlMjBDb2RlJTIwU2lnbmluZyUyMFBDQSUyMDIwMjEu
# Y3JsMIGuBggrBgEFBQcBAQSBoTCBnjBtBggrBgEFBQcwAoZhaHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBJRCUyMFZlcmlm
# aWVkJTIwQ29kZSUyMFNpZ25pbmclMjBQQ0ElMjAyMDIxLmNydDAtBggrBgEFBQcw
# AYYhaHR0cDovL29uZW9jc3AubWljcm9zb2Z0LmNvbS9vY3NwMA0GCSqGSIb3DQEB
# DAUAA4ICAQBnLThdlbMNIokdKtzSa8io+pEO95Cc3VOyY/hQsIIcdMyk2hJOzLt/
# M1WXfQyElDk/QtyLzX63TdOb5J+nO8t0pzzwi7ZYvMiNqKvAQO50sMOJn3T3hCPp
# pxNNhoGFVxz2UyiQ4b2vOrcsLK9TOEFXWbUMJObR9PM0wZsABIhu4k6VVLxEDe0G
# SeQX/ZE7PHfTg44Luft4IKqYmnv1Cuosp3glFYsVegLnMWZUZ8UtO9F8QCiAouJY
# hL5OlCksgDb9ve/HQhLFnelfg6dQubIFsqB9IlConYKJZ/HaMZvYtA7y9EORK4cx
# lvTetCXAHayiSXH0ueE/T92wVG0csv5VdUyj6yVrm22vlKYAkXINKvDOB8+s4h+T
# gShlUa2ACu2FWn7JzlTSbpk0IE8REuYmkuyE/BTkk93WDMx7PwLnn4J+5fkvbjjQ
# 08OewfpMhh8SuPdQKqmZ40I4W2UyJKMMTbet16JFimSqDChgnCB6lwlpe0gfbo97
# U7prpbfBKp6B2k2f7Y+TjWrQYN+OdcPOyQAdxGGPBwJSaJG3ohdklCxgAJ5anCxe
# Yl7SjQ5Eua6atjIeVhN0KfPLFPpYz5CQU+JC2H79x4d/O6YOFR9aYe54/CGup7dR
# UIfLSv1/j0DPc6Elf3YyWxloWj8yeY3kHrZFaAlRMwhAXyPQ3rEX9zCCB54wggWG
# oAMCAQICEzMAAAAHh6M0o3uljhwAAAAAAAcwDQYJKoZIhvcNAQEMBQAwdzELMAkG
# A1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjFIMEYGA1UE
# AxM/TWljcm9zb2Z0IElkZW50aXR5IFZlcmlmaWNhdGlvbiBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDIwMB4XDTIxMDQwMTIwMDUyMFoXDTM2MDQwMTIwMTUy
# MFowYzELMAkGA1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjE0MDIGA1UEAxMrTWljcm9zb2Z0IElEIFZlcmlmaWVkIENvZGUgU2lnbmluZyBQ
# Q0EgMjAyMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALLwwK8ZiCji
# 3VR6TElsaQhVCbRS/3pK+MHrJSj3Zxd3KU3rlfL3qrZilYKJNqztA9OQacr1AwoN
# cHbKBLbsQAhBnIB34zxf52bDpIO3NJlfIaTE/xrweLoQ71lzCHkD7A4As1Bs076I
# u+mA6cQzsYYH/Cbl1icwQ6C65rU4V9NQhNUwgrx9rGQ//h890Q8JdjLLw0nV+ayQ
# 2Fbkd242o9kH82RZsH3HEyqjAB5a8+Ae2nPIPc8sZU6ZE7iRrRZywRmrKDp5+Tcm
# JX9MRff241UaOBs4NmHOyke8oU1TYrkxh+YeHgfWo5tTgkoSMoayqoDpHOLJs+qG
# 8Tvh8SnifW2Jj3+ii11TS8/FGngEaNAWrbyfNrC69oKpRQXY9bGH6jn9NEJv9weF
# xhTwyvx9OJLXmRGbAUXN1U9nf4lXezky6Uh/cgjkVd6CGUAf0K+Jw+GE/5VpIVbc
# Nr9rNE50Sbmy/4RTCEGvOq3GhjITbCa4crCzTTHgYYjHs1NbOc6brH+eKpWLtr+b
# Gecy9CrwQyx7S/BfYJ+ozst7+yZtG2wR461uckFu0t+gCwLdN0A6cFtSRtR8bvxV
# FyWwTtgMMFRuBa3vmUOTnfKLsLefRaQcVTgRnzeLzdpt32cdYKp+dhr2ogc+qM6K
# 4CBI5/j4VFyC4QFeUP2YAidLtvpXRRo3AgMBAAGjggI1MIICMTAOBgNVHQ8BAf8E
# BAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNlBKbAPD2Ns72nX9c0p
# nqRIajDmMFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSME
# GDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYDVR0fBH0wezB5oHegdYZzaHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSWRl
# bnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIwQ2VydGlmaWNhdGUlMjBBdXRo
# b3JpdHklMjAyMDIwLmNybDCBwwYIKwYBBQUHAQEEgbYwgbMwgYEGCCsGAQUFBzAC
# hnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29m
# dCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIwUm9vdCUyMENlcnRpZmljYXRl
# JTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwLQYIKwYBBQUHMAGGIWh0dHA6Ly9vbmVv
# Y3NwLm1pY3Jvc29mdC5jb20vb2NzcDANBgkqhkiG9w0BAQwFAAOCAgEAfyUqnv7U
# q+rdZgrbVyNMul5skONbhls5fccPlmIbzi+OwVdPQ4H55v7VOInnmezQEeW4LqK0
# wja+fBznANbXLB0KrdMCbHQpbLvG6UA/Xv2pfpVIE1CRFfNF4XKO8XYEa3oW8oVH
# +KZHgIQRIwAbyFKQ9iyj4aOWeAzwk+f9E5StNp5T8FG7/VEURIVWArbAzPt9ThVN
# 3w1fAZkF7+YU9kbq1bCR2YD+MtunSQ1Rft6XG7b4e0ejRA7mB2IoX5hNh3UEauY0
# byxNRG+fT2MCEhQl9g2i2fs6VOG19CNep7SquKaBjhWmirYyANb0RJSLWjinMLXN
# OAga10n8i9jqeprzSMU5ODmrMCJE12xS/NWShg/tuLjAsKP6SzYZ+1Ry358ZTFcx
# 0FS/mx2vSoU8s8HRvy+rnXqyUJ9HBqS0DErVLjQwK8VtsBdekBmdTbQVoCgPCqr+
# PDPB3xajYnzevs7eidBsM71PINK2BoE2UfMwxCCX3mccFgx6UsQeRSdVVVNSyALQ
# e6PT12418xon2iDGE81OGCreLzDcMAZnrUAx4XQLUz6ZTl65yPUiOh3k7Yww94lD
# f+8oG2oZmDh5O1Qe38E+M3vhKwmzIeoB1dVLlz4i3IpaDcR+iuGjH2TdaC1ZOmBX
# iCRKJLj4DT2uhJ04ji+tHD6n58vhavFIrmcxghqUMIIakAIBATBxMFoxCzAJBgNV
# BAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMT
# Ik1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDICEzMABhp5Mx317l5t
# qaIAAAAGGnkwDQYJYIZIAWUDBAIBBQCgXjAQBgorBgEEAYI3AgEMMQIwADAZBgkq
# hkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAvBgkqhkiG9w0BCQQxIgQgJrZ0pb4hzyOS
# iVFJBgfOvPVz1GQoedAGu02OM3geVfowDQYJKoZIhvcNAQEBBQAEggGAdVspFuGt
# pFO4003KEt6i/pO9ju73dZNKUBU4LBzlUpuIgeks2aRxjDnDfOer4jfbrOlydqTJ
# 2oC0YwNUba93XgnQ97z6wYlASiEFNntUdWQLbFSRgJYNLECxgybWeyCj9VHqp5si
# t2iJU2YwKDLbkRsP8KrN+71hkMlA9m8ebiq9G0nFryAJ8pz//W+9Sz8blI3qbzJI
# 8W3dLwkj9eCZt7IQ7Y5gbCuZh6l1rKrkae36uApXIndkaozKSUgEf5nbMRUi91qe
# zYZCzjbrEIE7IVyNWK+d5WzfraPrfFTKzVoVDoGz9gnkMmUtYI/+1uFdKb+k7ErX
# xfWGjkIbzxkBSmL+jHnhRFg+s3YheWQobFXbyeTd8n7QtN5wOvqMG0RNhI1A05je
# 84VMEczIn0tKg+wl7x1xEkiVAsNU/OqwK7lexqEM/qoW5cWXxe7YDMjoC8ky/ZS7
# kYwRrkRw2p/KJ8aQypmnYIbKbc5Rr2eReKAVd3QE2Hu3ezJheD2gNVFcoYIYFDCC
# GBAGCisGAQQBgjcDAwExghgAMIIX/AYJKoZIhvcNAQcCoIIX7TCCF+kCAQMxDzAN
# BglghkgBZQMEAgEFADCCAWIGCyqGSIb3DQEJEAEEoIIBUQSCAU0wggFJAgEBBgor
# BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIKhT77yiqHfLfxm2PJ1ntcBE7d7O
# NgOeqsdWQYmKn+VRAgZpBeQSFFwYEzIwMjUxMTEyMTAzMjE1LjA3OVowBIACAfSg
# geGkgd4wgdsxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAj
# BgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJzAlBgNVBAsTHm5T
# aGllbGQgVFNTIEVTTjo3RDAwLTA1RTAtRDk0NzE1MDMGA1UEAxMsTWljcm9zb2Z0
# IFB1YmxpYyBSU0EgVGltZSBTdGFtcGluZyBBdXRob3JpdHmggg8hMIIHgjCCBWqg
# AwIBAgITMwAAAAXlzw//Zi7JhwAAAAAABTANBgkqhkiG9w0BAQwFADB3MQswCQYD
# VQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMUgwRgYDVQQD
# Ez9NaWNyb3NvZnQgSWRlbnRpdHkgVmVyaWZpY2F0aW9uIFJvb3QgQ2VydGlmaWNh
# dGUgQXV0aG9yaXR5IDIwMjAwHhcNMjAxMTE5MjAzMjMxWhcNMzUxMTE5MjA0MjMx
# WjBhMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MTIwMAYDVQQDEylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0Eg
# MjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJ5851Jj/eDFnwV9
# Y7UGIqMcHtfnlzPREwW9ZUZHd5HBXXBvf7KrQ5cMSqFSHGqg2/qJhYqOQxwuEQXG
# 8kB41wsDJP5d0zmLYKAY8Zxv3lYkuLDsfMuIEqvGYOPURAH+Ybl4SJEESnt0MbPE
# oKdNihwM5xGv0rGofJ1qOYSTNcc55EbBT7uq3wx3mXhtVmtcCEr5ZKTkKKE1CxZv
# NPWdGWJUPC6e4uRfWHIhZcgCsJ+sozf5EeH5KrlFnxpjKKTavwfFP6XaGZGWUG8T
# ZaiTogRoAlqcevbiqioUz1Yt4FRK53P6ovnUfANjIgM9JDdJ4e0qiDRm5sOTiEQt
# BLGd9Vhd1MadxoGcHrRCsS5rO9yhv2fjJHrmlQ0EIXmp4DhDBieKUGR+eZ4CNE3c
# tW4uvSDQVeSp9h1SaPV8UWEfyTxgGjOsRpeexIveR1MPTVf7gt8hY64XNPO6iyUG
# sEgt8c2PxF87E+CO7A28TpjNq5eLiiunhKbq0XbjkNoU5JhtYUrlmAbpxRjb9tSr
# eDdtACpm3rkpxp7AQndnI0Shu/fk1/rE3oWsDqMX3jjv40e8KN5YsJBnczyWB4Jy
# eeFMW3JBfdeAKhzohFe8U5w9WuvcP1E8cIxLoKSDzCCBOu0hWdjzKNu8Y5SwB1lt
# 5dQhABYyzR3dxEO/T1K/BVF3rV69AgMBAAGjggIbMIICFzAOBgNVHQ8BAf8EBAMC
# AYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFGtpKDo1L0hjQM972K9J6T7Z
# PdshMFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wEwYDVR0l
# BAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0T
# AQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYD
# VR0fBH0wezB5oHegdYZzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9j
# cmwvTWljcm9zb2Z0JTIwSWRlbnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIw
# Q2VydGlmaWNhdGUlMjBBdXRob3JpdHklMjAyMDIwLmNybDCBlAYIKwYBBQUHAQEE
# gYcwgYQwgYEGCCsGAQUFBzAChnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# b3BzL2NlcnRzL01pY3Jvc29mdCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIw
# Um9vdCUyMENlcnRpZmljYXRlJTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwDQYJKoZI
# hvcNAQEMBQADggIBAF+Idsd+bbVaFXXnTHho+k7h2ESZJRWluLE0Oa/pO+4ge/XE
# izXvhs0Y7+KVYyb4nHlugBesnFqBGEdC2IWmtKMyS1OWIviwpnK3aL5JedwzbeBF
# 7POyg6IGG/XhhJ3UqWeWTO+Czb1c2NP5zyEh89F72u9UIw+IfvM9lzDmc2O2END7
# MPnrcjWdQnrLn1Ntday7JSyrDvBdmgbNnCKNZPmhzoa8PccOiQljjTW6GePe5sGF
# uRHzdFt8y+bN2neF7Zu8hTO1I64XNGqst8S+w+RUdie8fXC1jKu3m9KGIqF4aldr
# YBamyh3g4nJPj/LR2CBaLyD+2BuGZCVmoNR/dSpRCxlot0i79dKOChmoONqbMI8m
# 04uLaEHAv4qwKHQ1vBzbV/nG89LDKbRSSvijmwJwxRxLLpMQ/u4xXxFfR4f/gksS
# kbJp7oqLwliDm/h+w0aJ/U5ccnYhYb7vPKNMN+SZDWycU5ODIRfyoGl59BsXR/Hp
# RGtiJquOYGmvA/pk5vC1lcnbeMrcWD/26ozePQ/TWfNXKBOmkFpvPE8CH+EeGGWz
# qTCjdAsno2jzTeNSxlx3glDGJgcdz5D/AAxw9Sdgq/+rY7jjgs7X6fqPTXPmaCAJ
# KVHAP19oEjJIBwD1LyHbaEgBxFCogYSOiUIr0Xqcr1nJfiWG2GwYe6ZoAF1bMIIH
# lzCCBX+gAwIBAgITMwAAAFXZ3WkmKPn44gAAAAAAVTANBgkqhkiG9w0BAQwFADBh
# MQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIw
# MAYDVQQDEylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAy
# MDAeFw0yNTEwMjMyMDQ2NDlaFw0yNjEwMjIyMDQ2NDlaMIHbMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046N0QwMC0w
# NUUwLUQ5NDcxNTAzBgNVBAMTLE1pY3Jvc29mdCBQdWJsaWMgUlNBIFRpbWUgU3Rh
# bXBpbmcgQXV0aG9yaXR5MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# vbkfkh5ZSLP0MCUWafaw/KZoVZu9iQx8r5JwhZvdrUi86UjCCFQONjQanrIxGF9h
# RGIZLQZ50gHrLC+4fpUEJff5t04VwByWC2/bWOuk6NmaTh9JpPZDcGzNR95Qlryj
# fEjtl+gxj12zNPEdADPplVfzt8cYRWFBx/Fbfch08k6P9p7jX2q1jFPbUxWYJ+xO
# yGC1aKhDGY5b+8wL39v6qC0HFIx/v3y+bep+aEXooK8VoeWK+szfaFjXo8YTcvQ8
# UL4szu9HFTuZNv6vvoJ7Ju+o5aTj51sph+0+FXW38TlL/rDBd5ia79jskLtOeHbD
# jkbljilwzegcxv9i49F05ZrS/5ELZCCY1VaqO7EOLKVaxxdAO5oy1vb0Bx0ZRVX1
# mxFjYzay2EC051k6yGJHm58y1oe2IKRa/SM1+BTGse6vHNi5Q2d5ZnoR9AOAUDDw
# JIIqRI4rZz2MSinh11WrXTG9urF2uoyd5Ve+8hxes9ABeP2PYQKlXYTAxvdaeanD
# TQ/vwmnM+yTcWzrVm84Z38XVFw4G7p/ZNZ2nscvv6uru2AevXcyV1t8ha7iWmhhg
# TWBNBrViuDlc3iPvOz2SVPbPeqhyY/NXwNZCAgc2H5pOztu6MwQxDIjte3XM/FkK
# BxHofS2abNT/0HG+xZtFqUJDaxgbJa6lN1zh7spjuQ8CAwEAAaOCAcswggHHMB0G
# A1UdDgQWBBRWBF8QbdwIA/DIv6nJFsrB16xltjAfBgNVHSMEGDAWgBRraSg6NS9I
# Y0DPe9ivSek+2T3bITBsBgNVHR8EZTBjMGGgX6BdhltodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBQdWJsaWMlMjBSU0ElMjBU
# aW1lc3RhbXBpbmclMjBDQSUyMDIwMjAuY3JsMHkGCCsGAQUFBwEBBG0wazBpBggr
# BgEFBQcwAoZdaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9N
# aWNyb3NvZnQlMjBQdWJsaWMlMjBSU0ElMjBUaW1lc3RhbXBpbmclMjBDQSUyMDIw
# MjAuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYD
# VR0PAQH/BAQDAgeAMGYGA1UdIARfMF0wUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYB
# BQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBv
# c2l0b3J5Lmh0bTAIBgZngQwBBAIwDQYJKoZIhvcNAQEMBQADggIBAFIe4ZJUe9qU
# KcWeWypchB58fXE/ZIWv2D5XP5/k/tB7LCN9BvmNSVKZ3VeclQM978wfEvuvdMQS
# Uv6Y20boIM8DK1K1IU9cP21MG0ExiHxaqjrikf2qbfrXIip4Ef3v2bNYKQxCxN3S
# czp1SX0H7uqK2L5OhfDEiXf15iou5hh+EPaaqp49czNQpJDOR/vfJghUc/qcslDP
# hoCZpZx8b2ODvywGQNXwqlbsmCS24uGmEkQ3UH5JUeN6c91yasVchS78riMrm6R9
# ZpAiO5pfNKMGU2MLm1A3pp098DcbFTAc95Hh6Qvkh//28F/Xe2bMFb6DL7Sw0ZO9
# 5v0gv0ZTyJfxS/LCxfraeEII9FSFOKAMEp1zNFSs2ue0GGjBt9yEEMUwvxq9ExFz
# 0aZzYm8ivJfffpIVDnX/+rVRTYcxIkQyFYslIhYlWF9SjCw5r49qakjMRNh8W9O7
# aaoolSVZleQZjGt0K8JzMlyp6hp2lbW6XqRx2cOHbbxJDxmENzohGUziI13lI2g2
# Bf5qibfC4bKNRpJo9lbE8HUbY0qJiE8u3SU8eDQaySPXOEhJjxRCQwwOvejYmBG5
# P7CckQNBSnnl12+FKRKgPoj0Mv+z5OMhj9z2MtpbnHLAkep0odQClEyyCG/uR5tK
# 5rW6mZH5Oq56UWS0NI6NV1JGS7Jri6jFMYIHRjCCB0ICAQEweDBhMQswCQYDVQQG
# EwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylN
# aWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMAITMwAAAFXZ
# 3WkmKPn44gAAAAAAVTANBglghkgBZQMEAgEFAKCCBJ8wEQYLKoZIhvcNAQkQAg8x
# AgUAMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcN
# MjUxMTEyMTAzMjE1WjAvBgkqhkiG9w0BCQQxIgQg1CgHKWRUmTKb/sjgKYHkNBEp
# jyf6vz1mzenzInO0nokwgbkGCyqGSIb3DQEJEAIvMYGpMIGmMIGjMIGgBCDYuTyX
# ZIZiu799/v4PaqsmeSzBxh0rqkYq7sYYavj+zTB8MGWkYzBhMQswCQYDVQQGEwJV
# UzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNy
# b3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMAITMwAAAFXZ3Wkm
# KPn44gAAAAAAVTCCA2EGCyqGSIb3DQEJEAISMYIDUDCCA0yhggNIMIIDRDCCAiwC
# AQEwggEJoYHhpIHeMIHbMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYD
# VQQLEx5uU2hpZWxkIFRTUyBFU046N0QwMC0wNUUwLUQ5NDcxNTAzBgNVBAMTLE1p
# Y3Jvc29mdCBQdWJsaWMgUlNBIFRpbWUgU3RhbXBpbmcgQXV0aG9yaXR5oiMKAQEw
# BwYFKw4DAhoDFQAdO1QBgmW/tuBZV5EGjhfsV4cN6qBnMGWkYzBhMQswCQYDVQQG
# EwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylN
# aWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMDANBgkqhkiG
# 9w0BAQsFAAIFAOy+OkMwIhgPMjAyNTExMTEyMjQyMTFaGA8yMDI1MTExMjIyNDIx
# MVowdzA9BgorBgEEAYRZCgQBMS8wLTAKAgUA7L46QwIBADAKAgEAAgIb/QIB/zAH
# AgEAAgISRjAKAgUA7L+LwwIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZ
# CgMCoAowCAIBAAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBCwUAA4IBAQCY
# MKF95oG1twqBO/NdsA4CGmVWo7z2bxWRHgei1drlpTPRhNR48RC86Qm6QzGF6SA1
# /mf3uKvptRyrsm/7gwz8EllXtVmTSZflF9vV8h3phVyvO2Ws+VKd5SIC6MbBh2dH
# fhKu7eWG00R2cZiluH3qkBCGkzvZVOaC4Y0T+s9ovX+9QbvxK6TgmS/e1zOEF1uC
# 3tf3bchw7IFCOBuIJ+CxhjConHb0fw1sfrzXmjBZO9fKu5v5poWC/M9gIZb9d85Z
# eqhhjVzmcDdctoRnk1aqFKYn+gN1IkAJt+CkSxHnCvHB6RNIsm2VIaI8LvG6hS4L
# qYnXbrvV93gg5kLzHHIZMA0GCSqGSIb3DQEBAQUABIICAJW1jz3N5/bnMEHtHmGh
# SFTRD7EdTtk6tRvvo/WN4Grxo6FtyYbM5m1Cyc8BdoOVhqnan+MnA62sHJVFSzV/
# 7ZTJj7W+L7zcI+VMcjdwKUGf+HtJyvADqOXVnl+9DHdMQMSKCyQs7JUsaW7iUQX5
# i/CjUmc/PjL8BuqKZTuBSFb5nL5NPwMXYFduoQ7zxoKjWPhwXb7vy3TeVr5N3DDb
# xUfYlvGt36xwU86QQTJMN6vzryMhs+aqwkeRrIe0KsJUCfG2Yiyf53DJQVallMK+
# ypYwkIfXVKYfqhH/+HeF3AswytUImxB/5Rs8yOKWvF0iYi4HYV2yMAVyznqL3HZX
# vI/F8Rq3dCeAnYUb/Jnj7ASZScPJRapCDoKDj//n9123YliINqba8Ln8ohStSeqS
# l5w+f3OFIJ6OrZpUJc+1wPvEVB4JypFYOY8G1GF7F9dW81KaRbXgnvkv0dqS7YUk
# q7ApsWyjBXZQEOZf1u02FZ108n3ZTkxz4MmoqP2uipWczLG7BTx47FdsKVF3XSRa
# YkwEUE64lBiVi9Ptu4eCShJ7o/9rLqt/VNosepF6VAuQI4ULKcTZi0qNakZ9xk9F
# tugUAZCjPWovt7ZamUoVAgZrf9JYxoH/06ToD0B4cm8wnSPwR1jtPh3OgD3SBgnZ
# JMqHKXA7ZR1tfRvgp29upncN
# SIG # End signature block
