param(
    [parameter(Mandatory = $false, Position = 0)][string] $ensureAdminGroup = $true,
    [parameter(Mandatory = $false, Position = 1)][string] $autologin = $true
)

# This script enables Assigned access and autologin on Windows 10

$username = "RadSight"
$application = "RadSight"
$toolsPath = "$env:SystemDrive\Tools\"
$sid = (Get-LocalUser -Name $username).SID.Value

# Get which application to update
$radsightVersion = (Get-AppPackage $application -AllUsers | Select-Object -ExpandProperty Version)
if (-not $radsightVersion) {
    $application = "RadDisplay"
    $radsightVersion = (Get-AppPackage $application -AllUsers | Select-Object -ExpandProperty Version)

    if (-not $radsightVersion) {
        $product = Get-WmiObject Win32_InstalledWin32Program -filter "Name LIKE '%RadSight App%'"
        if($product) {
            $application = "RadSight WPF"
        } else {
            $application = "RemoteDNA"
        }
    }
}

if ($ensureAdminGroup -eq "true") {
    # make sure we have an administrators group
    If ($Null -eq (Get-LocalGroup -Name "Administrators" -ErrorAction SilentlyContinue)) {
        # great administrators group
        New-LocalGroup -Name "Administrators"
        # make sure admin user is in it
        Add-LocalGroupMember -Group "Administrators" -Member $env:username
    }
}

# force user logoff
$sessionID = ((quser | Where-Object { $_ -match $username }) -split ' +')[2]
if ($null -ne $sessionID) {
    Write-Output "### Logging off kiosk user"

    ###logs off user
    Logoff $sessionID
}

Write-Output "### Enabling assigned access for user '$username' with application '$application'... ###"

if ($application -ne "RemoteDNA" -and $application -ne "RadSight WPF") {
    # set assigned access
    Set-AssignedAccess -AppName $application -UserName $username
}
else {
    # use edge
    & $toolsPath\Enable-Win32Kiosk.ps1
}

if ($autologin -eq "true") {
    # set autologin
    Write-Output "### Enabling auto login for user '$username'... ###"
    $RegPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
    Set-ItemProperty $RegPath "AutoAdminLogon" -Value "1" -type String 
    Set-ItemProperty $RegPath "DefaultUserName" -Value $username -type String
    Set-ItemProperty $RegPath "DefaultPassword" -Value "" -type String
}

Write-Output "### Enabling Complete. You can now close this window ###"
# SIG # Begin signature block
# MII+HQYJKoZIhvcNAQcCoII+DjCCPgoCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCClmbdbQmpm44NG
# gMbNTfXKWlsXZQJ9TURBxJfgCVoxRqCCIuAwggXMMIIDtKADAgECAhBUmNLR1FsZ
# lUgTecgRwIeZMA0GCSqGSIb3DQEBDAUAMHcxCzAJBgNVBAYTAlVTMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jvc29mdCBJZGVu
# dGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAy
# MDAeFw0yMDA0MTYxODM2MTZaFw00NTA0MTYxODQ0NDBaMHcxCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jv
# c29mdCBJZGVudGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRo
# b3JpdHkgMjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALORKgeD
# Bmf9np3gx8C3pOZCBH8Ppttf+9Va10Wg+3cL8IDzpm1aTXlT2KCGhFdFIMeiVPvH
# or+Kx24186IVxC9O40qFlkkN/76Z2BT2vCcH7kKbK/ULkgbk/WkTZaiRcvKYhOuD
# PQ7k13ESSCHLDe32R0m3m/nJxxe2hE//uKya13NnSYXjhr03QNAlhtTetcJtYmrV
# qXi8LW9J+eVsFBT9FMfTZRY33stuvF4pjf1imxUs1gXmuYkyM6Nix9fWUmcIxC70
# ViueC4fM7Ke0pqrrBc0ZV6U6CwQnHJFnni1iLS8evtrAIMsEGcoz+4m+mOJyoHI1
# vnnhnINv5G0Xb5DzPQCGdTiO0OBJmrvb0/gwytVXiGhNctO/bX9x2P29Da6SZEi3
# W295JrXNm5UhhNHvDzI9e1eM80UHTHzgXhgONXaLbZ7LNnSrBfjgc10yVpRnlyUK
# xjU9lJfnwUSLgP3B+PR0GeUw9gb7IVc+BhyLaxWGJ0l7gpPKWeh1R+g/OPTHU3mg
# trTiXFHvvV84wRPmeAyVWi7FQFkozA8kwOy6CXcjmTimthzax7ogttc32H83rwjj
# O3HbbnMbfZlysOSGM1l0tRYAe1BtxoYT2v3EOYI9JACaYNq6lMAFUSw0rFCZE4e7
# swWAsk0wAly4JoNdtGNz764jlU9gKL431VulAgMBAAGjVDBSMA4GA1UdDwEB/wQE
# AwIBhjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTIftJqhSobyhmYBAcnz1AQ
# T2ioojAQBgkrBgEEAYI3FQEEAwIBADANBgkqhkiG9w0BAQwFAAOCAgEAr2rd5hnn
# LZRDGU7L6VCVZKUDkQKL4jaAOxWiUsIWGbZqWl10QzD0m/9gdAmxIR6QFm3FJI9c
# Zohj9E/MffISTEAQiwGf2qnIrvKVG8+dBetJPnSgaFvlVixlHIJ+U9pW2UYXeZJF
# xBA2CFIpF8svpvJ+1Gkkih6PsHMNzBxKq7Kq7aeRYwFkIqgyuH4yKLNncy2RtNwx
# AQv3Rwqm8ddK7VZgxCwIo3tAsLx0J1KH1r6I3TeKiW5niB31yV2g/rarOoDXGpc8
# FzYiQR6sTdWD5jw4vU8w6VSp07YEwzJ2YbuwGMUrGLPAgNW3lbBeUU0i/OxYqujY
# lLSlLu2S3ucYfCFX3VVj979tzR/SpncocMfiWzpbCNJbTsgAlrPhgzavhgplXHT2
# 6ux6anSg8Evu75SjrFDyh+3XOjCDyft9V77l4/hByuVkrrOj7FjshZrM77nq81YY
# uVxzmq/FdxeDWds3GhhyVKVB0rYjdaNDmuV3fJZ5t0GNv+zcgKCf0Xd1WF81E+Al
# GmcLfc4l+gcK5GEh2NQc5QfGNpn0ltDGFf5Ozdeui53bFv0ExpK91IjmqaOqu/dk
# ODtfzAzQNb50GQOmxapMomE2gj4d8yu8l13bS3g7LfU772Aj6PXsCyM2la+YZr9T
# 03u4aUoqlmZpxJTG9F9urJh4iIAGXKKy7aIwggcEMIIE7KADAgECAhMzAAYaeTMd
# 9e5ebamiAAAABhp5MA0GCSqGSIb3DQEBDAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMTIk1pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIwHhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0
# MjAzMzAyWjCBgDELMAkGA1UEBhMCR0IxDzANBgNVBAgTBkRvcnNldDEOMAwGA1UE
# BxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0
# ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBEYXRhIE1hbmFnZW1lbnQgTHRkMIIBojAN
# BgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAmcc32FHGLV4FiKzse5eN9PjhviRa
# CblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK6e9cz2ayR0wdaIJSElN50pHTPrWMDQSL
# P7wkC1LovyoEhNzVb40CoJhCTXQTkHqnACMD9NqeDdyaF4r0Nb0u716+W67pv/CA
# CE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZVrqE61MYKeWlQWErF2VSZJaA8X9pJuvH
# 1WxVB+Mr7LIehjb9cxmUfelY1LkKimcVkdpSlZsBCZVANliqCugFrqMLJ1jQRmpd
# 8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp/hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8
# vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgXTHt5262ALFRyo3/NShidyL3k8a4hSFwD
# tWFHtXJv7HKGxbCCZdR8TGMFupw4IeWSBdApLGInTzSMhDxAyTK26McpCGRe3c1z
# BOw753ZIhx5k0LhWztmdCXWZIh9h1UBySzdDAgMBAAGjggIaMIICFjAMBgNVHRMB
# Af8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9BgNVHSUENjA0BgorBgEEAYI3YQEABggr
# BgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQU
# uRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYDVR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQW
# hCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZWaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSUQlMjBWZXJpZmllZCUyMENTJTIwQU9D
# JTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUFBwEBBIGYMIGVMGQGCCsGAQUFBzAChlho
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUy
# MElEJTIwVmVyaWZpZWQlMjBDUyUyMEFPQyUyMENBJTIwMDIuY3J0MC0GCCsGAQUF
# BzABhiFodHRwOi8vb25lb2NzcC5taWNyb3NvZnQuY29tL29jc3AwZgYDVR0gBF8w
# XTBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMAgGBmeBDAEEATAN
# BgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKMPRNLuXymFInLM6+tzuw0G5i8qcvMna7j
# ltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYcn19Kvk5bdbRQW50EmDEdH5ybIuE1v5VP
# OjGAg9xvXFk1RdISpST9ghRE+JpdI2gx1uKGUGFCjll6nU/WuFpskg2jV6HQVg7t
# cdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBGDrfJJwLty0ageLObO88+fJdES7ZYsmC1
# Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9yQ+UTijQVYml/vi4oOLROYVk9Auf31WB5
# qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U++cDFVsMH6hvGXrByof7kuE6BQbYk2T0
# 8MaNus4xleApIAYyWr8Prtwe71+T2S+OM/NE+T0JwATBrSwUjBftUzM9nQ3iHGJW
# gT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcbQmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1
# Xjau+kf/EJeYzrhw1H5++1xVbSL1a+BudVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5o
# ERoxq92UpXtQYMo8AOC0TqmdFsqrAmmqi+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJ
# wyGaI2eSqCeROzqTE/BUP6ICU26FD03Mr1QnmMVAN3ligIAAAHHTMJmNaLHVuhVU
# DCAwggcEMIIE7KADAgECAhMzAAYaeTMd9e5ebamiAAAABhp5MA0GCSqGSIb3DQEB
# DAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xKzApBgNVBAMTIk1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIw
# HhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0MjAzMzAyWjCBgDELMAkGA1UEBhMCR0Ix
# DzANBgNVBAgTBkRvcnNldDEOMAwGA1UEBxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0
# aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBE
# YXRhIE1hbmFnZW1lbnQgTHRkMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC
# AYEAmcc32FHGLV4FiKzse5eN9PjhviRaCblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK
# 6e9cz2ayR0wdaIJSElN50pHTPrWMDQSLP7wkC1LovyoEhNzVb40CoJhCTXQTkHqn
# ACMD9NqeDdyaF4r0Nb0u716+W67pv/CACE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZ
# VrqE61MYKeWlQWErF2VSZJaA8X9pJuvH1WxVB+Mr7LIehjb9cxmUfelY1LkKimcV
# kdpSlZsBCZVANliqCugFrqMLJ1jQRmpd8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp
# /hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgX
# THt5262ALFRyo3/NShidyL3k8a4hSFwDtWFHtXJv7HKGxbCCZdR8TGMFupw4IeWS
# BdApLGInTzSMhDxAyTK26McpCGRe3c1zBOw753ZIhx5k0LhWztmdCXWZIh9h1UBy
# SzdDAgMBAAGjggIaMIICFjAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9
# BgNVHSUENjA0BgorBgEEAYI3YQEABggrBgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw
# 4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQUuRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYD
# VR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQWhCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZW
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIw
# SUQlMjBWZXJpZmllZCUyMENTJTIwQU9DJTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUF
# BwEBBIGYMIGVMGQGCCsGAQUFBzAChlhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMElEJTIwVmVyaWZpZWQlMjBDUyUyMEFP
# QyUyMENBJTIwMDIuY3J0MC0GCCsGAQUFBzABhiFodHRwOi8vb25lb2NzcC5taWNy
# b3NvZnQuY29tL29jc3AwZgYDVR0gBF8wXTBRBgwrBgEEAYI3TIN9AQEwQTA/Bggr
# BgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9Eb2NzL1Jl
# cG9zaXRvcnkuaHRtMAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKM
# PRNLuXymFInLM6+tzuw0G5i8qcvMna7jltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYc
# n19Kvk5bdbRQW50EmDEdH5ybIuE1v5VPOjGAg9xvXFk1RdISpST9ghRE+JpdI2gx
# 1uKGUGFCjll6nU/WuFpskg2jV6HQVg7tcdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBG
# DrfJJwLty0ageLObO88+fJdES7ZYsmC1Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9y
# Q+UTijQVYml/vi4oOLROYVk9Auf31WB5qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U
# ++cDFVsMH6hvGXrByof7kuE6BQbYk2T08MaNus4xleApIAYyWr8Prtwe71+T2S+O
# M/NE+T0JwATBrSwUjBftUzM9nQ3iHGJWgT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcb
# QmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1Xjau+kf/EJeYzrhw1H5++1xVbSL1a+Bu
# dVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5oERoxq92UpXtQYMo8AOC0TqmdFsqrAmmq
# i+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJwyGaI2eSqCeROzqTE/BUP6ICU26FD03M
# r1QnmMVAN3ligIAAAHHTMJmNaLHVuhVUDCAwggdaMIIFQqADAgECAhMzAAAABJZQ
# S9Lb7suIAAAAAAAEMA0GCSqGSIb3DQEBDAUAMGMxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xNDAyBgNVBAMTK01pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDb2RlIFNpZ25pbmcgUENBIDIwMjEwHhcNMjEwNDEzMTczMTUy
# WhcNMjYwNDEzMTczMTUyWjBaMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSswKQYDVQQDEyJNaWNyb3NvZnQgSUQgVmVyaWZpZWQg
# Q1MgQU9DIENBIDAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA4c6g
# 6DOiY6bAOwCPbBlQF2tjo3ckUZuab5ZorMnRp4rOmwZDiTbIpzFkZ/k8k4ivBJV1
# w5/b/oykI+eXAqaaxMdyAO0ModnEW7InfQ+rTkykEzHxRbCNg6KDsTnYc/YdL7II
# iJli8k51upaHLL7CYm9YNc0SFYvlaFj2O0HjO9y/NRmcWNjamZOlRjxW2cWgUsUd
# azSHgRCek87V2bM/17b+o8WXUW91IpggRasmiZ65WEFHXKbyhm2LbhBK6ZWmQoFe
# E+GWrKWCGK/q/4RiTaMNhHXWvWv+//I58UtOxVi3DaK1fQ6YLyIIGHzD4CmtcrGi
# vxupq/crrHunGNB7//Qmul2ZP9HcOmY/aptgUnwT+20g/A37iDfuuVw6yS2Lo0/k
# p/jb+J8vE4FMqIiwxGByL482PMVBC3qd/NbFQa8Mmj6ensU+HEqv9ar+AbcKwumb
# ZqJJKmQrGaSNdWfk2NodgcWOmq7jyhbxwZOjnLj0/bwnsUNcNAe09v+qiozyQQes
# 8A3UXPcRQb8G+c0yaO2ICifWTK7ySuyUJ88k1mtN22CNftbjitiAeafoZ9Vmhn5R
# fb+S/K5arVvTcLukt5PdTDQxl557EIE6A+6XFBpdsjOzkLzdEh7ELk8PVPMjQfPC
# gKtJ84c17fd2C9+pxF1lEQUFXY/YtCL+Nms9cWUCAwEAAaOCAg4wggIKMA4GA1Ud
# DwEB/wQEAwIBhjAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUJEWZoXeQKnzD
# yoOwbmQWhCr4LGcwVAYDVR0gBE0wSzBJBgRVHSAAMEEwPwYIKwYBBQUHAgEWM2h0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0
# bTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTASBgNVHRMBAf8ECDAGAQH/AgEA
# MB8GA1UdIwQYMBaAFNlBKbAPD2Ns72nX9c0pnqRIajDmMHAGA1UdHwRpMGcwZaBj
# oGGGX2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29m
# dCUyMElEJTIwVmVyaWZpZWQlMjBDb2RlJTIwU2lnbmluZyUyMFBDQSUyMDIwMjEu
# Y3JsMIGuBggrBgEFBQcBAQSBoTCBnjBtBggrBgEFBQcwAoZhaHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBJRCUyMFZlcmlm
# aWVkJTIwQ29kZSUyMFNpZ25pbmclMjBQQ0ElMjAyMDIxLmNydDAtBggrBgEFBQcw
# AYYhaHR0cDovL29uZW9jc3AubWljcm9zb2Z0LmNvbS9vY3NwMA0GCSqGSIb3DQEB
# DAUAA4ICAQBnLThdlbMNIokdKtzSa8io+pEO95Cc3VOyY/hQsIIcdMyk2hJOzLt/
# M1WXfQyElDk/QtyLzX63TdOb5J+nO8t0pzzwi7ZYvMiNqKvAQO50sMOJn3T3hCPp
# pxNNhoGFVxz2UyiQ4b2vOrcsLK9TOEFXWbUMJObR9PM0wZsABIhu4k6VVLxEDe0G
# SeQX/ZE7PHfTg44Luft4IKqYmnv1Cuosp3glFYsVegLnMWZUZ8UtO9F8QCiAouJY
# hL5OlCksgDb9ve/HQhLFnelfg6dQubIFsqB9IlConYKJZ/HaMZvYtA7y9EORK4cx
# lvTetCXAHayiSXH0ueE/T92wVG0csv5VdUyj6yVrm22vlKYAkXINKvDOB8+s4h+T
# gShlUa2ACu2FWn7JzlTSbpk0IE8REuYmkuyE/BTkk93WDMx7PwLnn4J+5fkvbjjQ
# 08OewfpMhh8SuPdQKqmZ40I4W2UyJKMMTbet16JFimSqDChgnCB6lwlpe0gfbo97
# U7prpbfBKp6B2k2f7Y+TjWrQYN+OdcPOyQAdxGGPBwJSaJG3ohdklCxgAJ5anCxe
# Yl7SjQ5Eua6atjIeVhN0KfPLFPpYz5CQU+JC2H79x4d/O6YOFR9aYe54/CGup7dR
# UIfLSv1/j0DPc6Elf3YyWxloWj8yeY3kHrZFaAlRMwhAXyPQ3rEX9zCCB54wggWG
# oAMCAQICEzMAAAAHh6M0o3uljhwAAAAAAAcwDQYJKoZIhvcNAQEMBQAwdzELMAkG
# A1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjFIMEYGA1UE
# AxM/TWljcm9zb2Z0IElkZW50aXR5IFZlcmlmaWNhdGlvbiBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDIwMB4XDTIxMDQwMTIwMDUyMFoXDTM2MDQwMTIwMTUy
# MFowYzELMAkGA1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjE0MDIGA1UEAxMrTWljcm9zb2Z0IElEIFZlcmlmaWVkIENvZGUgU2lnbmluZyBQ
# Q0EgMjAyMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALLwwK8ZiCji
# 3VR6TElsaQhVCbRS/3pK+MHrJSj3Zxd3KU3rlfL3qrZilYKJNqztA9OQacr1AwoN
# cHbKBLbsQAhBnIB34zxf52bDpIO3NJlfIaTE/xrweLoQ71lzCHkD7A4As1Bs076I
# u+mA6cQzsYYH/Cbl1icwQ6C65rU4V9NQhNUwgrx9rGQ//h890Q8JdjLLw0nV+ayQ
# 2Fbkd242o9kH82RZsH3HEyqjAB5a8+Ae2nPIPc8sZU6ZE7iRrRZywRmrKDp5+Tcm
# JX9MRff241UaOBs4NmHOyke8oU1TYrkxh+YeHgfWo5tTgkoSMoayqoDpHOLJs+qG
# 8Tvh8SnifW2Jj3+ii11TS8/FGngEaNAWrbyfNrC69oKpRQXY9bGH6jn9NEJv9weF
# xhTwyvx9OJLXmRGbAUXN1U9nf4lXezky6Uh/cgjkVd6CGUAf0K+Jw+GE/5VpIVbc
# Nr9rNE50Sbmy/4RTCEGvOq3GhjITbCa4crCzTTHgYYjHs1NbOc6brH+eKpWLtr+b
# Gecy9CrwQyx7S/BfYJ+ozst7+yZtG2wR461uckFu0t+gCwLdN0A6cFtSRtR8bvxV
# FyWwTtgMMFRuBa3vmUOTnfKLsLefRaQcVTgRnzeLzdpt32cdYKp+dhr2ogc+qM6K
# 4CBI5/j4VFyC4QFeUP2YAidLtvpXRRo3AgMBAAGjggI1MIICMTAOBgNVHQ8BAf8E
# BAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNlBKbAPD2Ns72nX9c0p
# nqRIajDmMFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSME
# GDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYDVR0fBH0wezB5oHegdYZzaHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSWRl
# bnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIwQ2VydGlmaWNhdGUlMjBBdXRo
# b3JpdHklMjAyMDIwLmNybDCBwwYIKwYBBQUHAQEEgbYwgbMwgYEGCCsGAQUFBzAC
# hnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29m
# dCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIwUm9vdCUyMENlcnRpZmljYXRl
# JTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwLQYIKwYBBQUHMAGGIWh0dHA6Ly9vbmVv
# Y3NwLm1pY3Jvc29mdC5jb20vb2NzcDANBgkqhkiG9w0BAQwFAAOCAgEAfyUqnv7U
# q+rdZgrbVyNMul5skONbhls5fccPlmIbzi+OwVdPQ4H55v7VOInnmezQEeW4LqK0
# wja+fBznANbXLB0KrdMCbHQpbLvG6UA/Xv2pfpVIE1CRFfNF4XKO8XYEa3oW8oVH
# +KZHgIQRIwAbyFKQ9iyj4aOWeAzwk+f9E5StNp5T8FG7/VEURIVWArbAzPt9ThVN
# 3w1fAZkF7+YU9kbq1bCR2YD+MtunSQ1Rft6XG7b4e0ejRA7mB2IoX5hNh3UEauY0
# byxNRG+fT2MCEhQl9g2i2fs6VOG19CNep7SquKaBjhWmirYyANb0RJSLWjinMLXN
# OAga10n8i9jqeprzSMU5ODmrMCJE12xS/NWShg/tuLjAsKP6SzYZ+1Ry358ZTFcx
# 0FS/mx2vSoU8s8HRvy+rnXqyUJ9HBqS0DErVLjQwK8VtsBdekBmdTbQVoCgPCqr+
# PDPB3xajYnzevs7eidBsM71PINK2BoE2UfMwxCCX3mccFgx6UsQeRSdVVVNSyALQ
# e6PT12418xon2iDGE81OGCreLzDcMAZnrUAx4XQLUz6ZTl65yPUiOh3k7Yww94lD
# f+8oG2oZmDh5O1Qe38E+M3vhKwmzIeoB1dVLlz4i3IpaDcR+iuGjH2TdaC1ZOmBX
# iCRKJLj4DT2uhJ04ji+tHD6n58vhavFIrmcxghqTMIIajwIBATBxMFoxCzAJBgNV
# BAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMT
# Ik1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDICEzMABhp5Mx317l5t
# qaIAAAAGGnkwDQYJYIZIAWUDBAIBBQCgXjAQBgorBgEEAYI3AgEMMQIwADAZBgkq
# hkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAvBgkqhkiG9w0BCQQxIgQgPyxGHoQZpL6/
# uIRR30Mx/SJ8B+Cz+oQ1VJTR4cp1Yy0wDQYJKoZIhvcNAQEBBQAEggGAQ/CezE9w
# Ad0OHLrjKTXx9CWZc4Xx9Ii6xvy8/a0RByQYDWassj5UF6AsAlXzBeX4SbGH4BJg
# fT+ZEVgek98M14oKHH3Ka1rqETC9CMSXtLGLa7A2f9pLQog3DKdQXfsYTPMx4OUB
# krQEZtc/1ZRCQ3R0fPLC9LkBsuSvt9vszxH3RkCh7Njg/J4TEhzMyAMpNL+UctHK
# oAS4pC3dc5jHIC9BmdFnVs0SlOIJM/RrT4OUdX5baW3+/voI5xPCJDETOqu1Znmg
# QB19uv3OV7PmVzhiOvMqtMKG1UvfTJByp1KS47O8P2FfaMPkRVmaTCWMVymFogEQ
# gSqFOQzqrAgIyJTfka+4QglFx49f2sX6gHIBi9GGj4fzhuRXS70EDhybkI/H5yru
# g/ksS/6OPL2vaw7HULgv5VwpVhYA11eIXE4mEZUd84LEblBRLBRbAhpO/07NQjuj
# ghejxU8X6EY8IlNpncFXiifiIIzNfO1ZkTJM2WceI7OrexpTXSXvQi+8oYIYEzCC
# GA8GCisGAQQBgjcDAwExghf/MIIX+wYJKoZIhvcNAQcCoIIX7DCCF+gCAQMxDzAN
# BglghkgBZQMEAgEFADCCAWEGCyqGSIb3DQEJEAEEoIIBUASCAUwwggFIAgEBBgor
# BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIMQBCBZkZua8GXG+3jfcz8+t2h6K
# gvI6b2SFyqn4cNMtAgZpBeQSFJUYEjIwMjUxMTEyMTAzMjI4LjA3WjAEgAIB9KCB
# 4aSB3jCB2zELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjElMCMG
# A1UECxMcTWljcm9zb2Z0IEFtZXJpY2EgT3BlcmF0aW9uczEnMCUGA1UECxMeblNo
# aWVsZCBUU1MgRVNOOjdEMDAtMDVFMC1EOTQ3MTUwMwYDVQQDEyxNaWNyb3NvZnQg
# UHVibGljIFJTQSBUaW1lIFN0YW1waW5nIEF1dGhvcml0eaCCDyEwggeCMIIFaqAD
# AgECAhMzAAAABeXPD/9mLsmHAAAAAAAFMA0GCSqGSIb3DQEBDAUAMHcxCzAJBgNV
# BAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMT
# P01pY3Jvc29mdCBJZGVudGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAyMDAeFw0yMDExMTkyMDMyMzFaFw0zNTExMTkyMDQyMzFa
# MGExCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# MjAwBgNVBAMTKU1pY3Jvc29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAy
# MDIwMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAnnznUmP94MWfBX1j
# tQYioxwe1+eXM9ETBb1lRkd3kcFdcG9/sqtDlwxKoVIcaqDb+omFio5DHC4RBcby
# QHjXCwMk/l3TOYtgoBjxnG/eViS4sOx8y4gSq8Zg49REAf5huXhIkQRKe3Qxs8Sg
# p02KHAznEa/Ssah8nWo5hJM1xznkRsFPu6rfDHeZeG1Wa1wISvlkpOQooTULFm80
# 9Z0ZYlQ8Lp7i5F9YciFlyAKwn6yjN/kR4fkquUWfGmMopNq/B8U/pdoZkZZQbxNl
# qJOiBGgCWpx69uKqKhTPVi3gVErnc/qi+dR8A2MiAz0kN0nh7SqINGbmw5OIRC0E
# sZ31WF3Uxp3GgZwetEKxLms73KG/Z+MkeuaVDQQheangOEMGJ4pQZH55ngI0Tdy1
# bi69INBV5Kn2HVJo9XxRYR/JPGAaM6xGl57Ei95HUw9NV/uC3yFjrhc087qLJQaw
# SC3xzY/EXzsT4I7sDbxOmM2rl4uKK6eEpurRduOQ2hTkmG1hSuWYBunFGNv21Kt4
# N20AKmbeuSnGnsBCd2cjRKG79+TX+sTehawOoxfeOO/jR7wo3liwkGdzPJYHgnJ5
# 4UxbckF914AqHOiEV7xTnD1a69w/UTxwjEugpIPMIIE67SFZ2PMo27xjlLAHWW3l
# 1CEAFjLNHd3EQ79PUr8FUXetXr0CAwEAAaOCAhswggIXMA4GA1UdDwEB/wQEAwIB
# hjAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUa2koOjUvSGNAz3vYr0npPtk9
# 2yEwVAYDVR0gBE0wSzBJBgRVHSAAMEEwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0bTATBgNVHSUE
# DDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTAPBgNVHRMB
# Af8EBTADAQH/MB8GA1UdIwQYMBaAFMh+0mqFKhvKGZgEByfPUBBPaKiiMIGEBgNV
# HR8EfTB7MHmgd6B1hnNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2Ny
# bC9NaWNyb3NvZnQlMjBJZGVudGl0eSUyMFZlcmlmaWNhdGlvbiUyMFJvb3QlMjBD
# ZXJ0aWZpY2F0ZSUyMEF1dGhvcml0eSUyMDIwMjAuY3JsMIGUBggrBgEFBQcBAQSB
# hzCBhDCBgQYIKwYBBQUHMAKGdWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lv
# cHMvY2VydHMvTWljcm9zb2Z0JTIwSWRlbnRpdHklMjBWZXJpZmljYXRpb24lMjBS
# b290JTIwQ2VydGlmaWNhdGUlMjBBdXRob3JpdHklMjAyMDIwLmNydDANBgkqhkiG
# 9w0BAQwFAAOCAgEAX4h2x35ttVoVdedMeGj6TuHYRJklFaW4sTQ5r+k77iB79cSL
# Ne+GzRjv4pVjJviceW6AF6ycWoEYR0LYhaa0ozJLU5Yi+LCmcrdovkl53DNt4EXs
# 87KDogYb9eGEndSpZ5ZM74LNvVzY0/nPISHz0Xva71QjD4h+8z2XMOZzY7YQ0Psw
# +etyNZ1CesufU211rLslLKsO8F2aBs2cIo1k+aHOhrw9xw6JCWONNboZ497mwYW5
# EfN0W3zL5s3ad4Xtm7yFM7Ujrhc0aqy3xL7D5FR2J7x9cLWMq7eb0oYioXhqV2tg
# FqbKHeDick+P8tHYIFovIP7YG4ZkJWag1H91KlELGWi3SLv10o4KGag42pswjybT
# i4toQcC/irAodDW8HNtX+cbz0sMptFJK+KObAnDFHEsukxD+7jFfEV9Hh/+CSxKR
# smnuiovCWIOb+H7DRon9TlxydiFhvu88o0w35JkNbJxTk4MhF/KgaXn0GxdH8elE
# a2Imq45gaa8D+mTm8LWVydt4ytxYP/bqjN49D9NZ81coE6aQWm88TwIf4R4YZbOp
# MKN0CyejaPNN41LGXHeCUMYmBx3PkP8ADHD1J2Cr/6tjuOOCztfp+o9Nc+ZoIAkp
# UcA/X2gSMkgHAPUvIdtoSAHEUKiBhI6JQivRepyvWcl+JYbYbBh7pmgAXVswggeX
# MIIFf6ADAgECAhMzAAAAVdndaSYo+fjiAAAAAABVMA0GCSqGSIb3DQEBDAUAMGEx
# CzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAw
# BgNVBAMTKU1pY3Jvc29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIw
# MB4XDTI1MTAyMzIwNDY0OVoXDTI2MTAyMjIwNDY0OVowgdsxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVy
# aWNhIE9wZXJhdGlvbnMxJzAlBgNVBAsTHm5TaGllbGQgVFNTIEVTTjo3RDAwLTA1
# RTAtRDk0NzE1MDMGA1UEAxMsTWljcm9zb2Z0IFB1YmxpYyBSU0EgVGltZSBTdGFt
# cGluZyBBdXRob3JpdHkwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC9
# uR+SHllIs/QwJRZp9rD8pmhVm72JDHyvknCFm92tSLzpSMIIVA42NBqesjEYX2FE
# YhktBnnSAessL7h+lQQl9/m3ThXAHJYLb9tY66To2ZpOH0mk9kNwbM1H3lCWvKN8
# SO2X6DGPXbM08R0AM+mVV/O3xxhFYUHH8Vt9yHTyTo/2nuNfarWMU9tTFZgn7E7I
# YLVoqEMZjlv7zAvf2/qoLQcUjH+/fL5t6n5oReigrxWh5Yr6zN9oWNejxhNy9DxQ
# vizO70cVO5k2/q++gnsm76jlpOPnWymH7T4VdbfxOUv+sMF3mJrv2OyQu054dsOO
# RuWOKXDN6BzG/2Lj0XTlmtL/kQtkIJjVVqo7sQ4spVrHF0A7mjLW9vQHHRlFVfWb
# EWNjNrLYQLTnWTrIYkebnzLWh7YgpFr9IzX4FMax7q8c2LlDZ3lmehH0A4BQMPAk
# gipEjitnPYxKKeHXVatdMb26sXa6jJ3lV77yHF6z0AF4/Y9hAqVdhMDG91p5qcNN
# D+/Cacz7JNxbOtWbzhnfxdUXDgbun9k1naexy+/q6u7YB69dzJXW3yFruJaaGGBN
# YE0GtWK4OVzeI+87PZJU9s96qHJj81fA1kICBzYfmk7O27ozBDEMiO17dcz8WQoH
# Eeh9LZps1P/Qcb7Fm0WpQkNrGBslrqU3XOHuymO5DwIDAQABo4IByzCCAccwHQYD
# VR0OBBYEFFYEXxBt3AgD8Mi/qckWysHXrGW2MB8GA1UdIwQYMBaAFGtpKDo1L0hj
# QM972K9J6T7ZPdshMGwGA1UdHwRlMGMwYaBfoF2GW2h0dHA6Ly93d3cubWljcm9z
# b2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29mdCUyMFB1YmxpYyUyMFJTQSUyMFRp
# bWVzdGFtcGluZyUyMENBJTIwMjAyMC5jcmwweQYIKwYBBQUHAQEEbTBrMGkGCCsG
# AQUFBzAChl1odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01p
# Y3Jvc29mdCUyMFB1YmxpYyUyMFJTQSUyMFRpbWVzdGFtcGluZyUyMENBJTIwMjAy
# MC5jcnQwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAOBgNV
# HQ8BAf8EBAMCB4AwZgYDVR0gBF8wXTBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEF
# BQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9z
# aXRvcnkuaHRtMAgGBmeBDAEEAjANBgkqhkiG9w0BAQwFAAOCAgEAUh7hklR72pQp
# xZ5bKlyEHnx9cT9kha/YPlc/n+T+0HssI30G+Y1JUpndV5yVAz3vzB8S+690xBJS
# /pjbRuggzwMrUrUhT1w/bUwbQTGIfFqqOuKR/apt+tciKngR/e/Zs1gpDELE3dJz
# OnVJfQfu6orYvk6F8MSJd/XmKi7mGH4Q9pqqnj1zM1CkkM5H+98mCFRz+pyyUM+G
# gJmlnHxvY4O/LAZA1fCqVuyYJLbi4aYSRDdQfklR43pz3XJqxVyFLvyuIyubpH1m
# kCI7ml80owZTYwubUDemnT3wNxsVMBz3keHpC+SH//bwX9d7ZswVvoMvtLDRk73m
# /SC/RlPIl/FL8sLF+tp4Qgj0VIU4oAwSnXM0VKza57QYaMG33IQQxTC/Gr0TEXPR
# pnNibyK8l99+khUOdf/6tVFNhzEiRDIViyUiFiVYX1KMLDmvj2pqSMxE2Hxb07tp
# qiiVJVmV5BmMa3QrwnMyXKnqGnaVtbpepHHZw4dtvEkPGYQ3OiEZTOIjXeUjaDYF
# /mqJt8Lhso1Gkmj2VsTwdRtjSomITy7dJTx4NBrJI9c4SEmPFEJDDA696NiYEbk/
# sJyRA0FKeeXXb4UpEqA+iPQy/7Pk4yGP3PYy2luccsCR6nSh1AKUTLIIb+5Hm0rm
# tbqZkfk6rnpRZLQ0jo1XUkZLsmuLqMUxggdGMIIHQgIBATB4MGExCzAJBgNVBAYT
# AlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1p
# Y3Jvc29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIwAhMzAAAAVdnd
# aSYo+fjiAAAAAABVMA0GCWCGSAFlAwQCAQUAoIIEnzARBgsqhkiG9w0BCRACDzEC
# BQAwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0y
# NTExMTIxMDMyMjhaMC8GCSqGSIb3DQEJBDEiBCDFC2M3UeZD2ESUh1nWcyUqcHBj
# J3FgmyJO8kfPsfjOITCBuQYLKoZIhvcNAQkQAi8xgakwgaYwgaMwgaAEINi5PJdk
# hmK7v33+/g9qqyZ5LMHGHSuqRiruxhhq+P7NMHwwZaRjMGExCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jv
# c29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIwAhMzAAAAVdndaSYo
# +fjiAAAAAABVMIIDYQYLKoZIhvcNAQkQAhIxggNQMIIDTKGCA0gwggNEMIICLAIB
# ATCCAQmhgeGkgd4wgdsxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJTAjBgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJzAlBgNV
# BAsTHm5TaGllbGQgVFNTIEVTTjo3RDAwLTA1RTAtRDk0NzE1MDMGA1UEAxMsTWlj
# cm9zb2Z0IFB1YmxpYyBSU0EgVGltZSBTdGFtcGluZyBBdXRob3JpdHmiIwoBATAH
# BgUrDgMCGgMVAB07VAGCZb+24FlXkQaOF+xXhw3qoGcwZaRjMGExCzAJBgNVBAYT
# AlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1p
# Y3Jvc29mdCBQdWJsaWMgUlNBIFRpbWVzdGFtcGluZyBDQSAyMDIwMA0GCSqGSIb3
# DQEBCwUAAgUA7L46QzAiGA8yMDI1MTExMTIyNDIxMVoYDzIwMjUxMTEyMjI0MjEx
# WjB3MD0GCisGAQQBhFkKBAExLzAtMAoCBQDsvjpDAgEAMAoCAQACAhv9AgH/MAcC
# AQACAhJGMAoCBQDsv4vDAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkK
# AwKgCjAIAgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAJgw
# oX3mgbW3CoE7812wDgIaZVajvPZvFZEeB6LV2uWlM9GE1HjxELzpCbpDMYXpIDX+
# Z/e4q+m1HKuyb/uDDPwSWVe1WZNJl+UX29XyHemFXK87Zaz5Up3lIgLoxsGHZ0d+
# Eq7t5YbTRHZxmKW4feqQEIaTO9lU5oLhjRP6z2i9f71Bu/ErpOCZL97XM4QXW4Le
# 1/dtyHDsgUI4G4gn4LGGMKicdvR/DWx+vNeaMFk718q7m/mmhYL8z2Ahlv13zll6
# qGGNXOZwN1y2hGeTVqoUpif6A3UiQAm34KRLEecK8cHpE0iybZUhojwu8bqFLgup
# idduu9X3eCDmQvMcchkwDQYJKoZIhvcNAQEBBQAEggIAcjI7gVxed8nldvwM64pu
# LMwQcCtCptc7r5YS7ZW4cBXawtuAGL9UxY7d6zNtO5K0Mf8idxQVSWrakII24LkC
# JEfdI16nWHRZyB+BWXyGtYjZXY3Cz0fmN6SbA5Kt3TaWIZJN9nI86AmcQHgXA+GR
# FhJdMf4KCnQHNCbsFbBkZ++OQCNnUqr2MmNjoiwSV+sYPdKDz2Lm8VR2NrYBcn4m
# n3jK/EzhTelGF4KXJ81osn3QlE+iPm5Q8p96XMZ4kSvBHGFSzDdJzptfklgcCgqc
# L0XFZdRgnz+c5CrIVL884VVDPXxxLWuEp8KgdrBELYlgGVSZ88460/1WdOXS8MoZ
# yYfr1C2IN82FreVqf7t3K08obhF+wIFX+qaTgykToDxrtIvdsevzuYC8heW43kdT
# R5SbEpozc+Bqty9cv+oqwdA/kD3+/7EJmFeA+2cIj2Mqz2HxTts+Ccma8tf+8Cxa
# lhtZuFNXpAAcTJSlsgEGvTQiBRB8HHv/p5+SDHuSQ0Pr4xQParkSOD9OMkJZmmXp
# q1AW4XmUeskPEzacQz+2g+kX+w5TOkdUfv4cAtpOH0YqPg51nz6DPqd6gTzCS2UJ
# C2w4UGGP30QQoUj4LQYKUERk46x1jLEaHLu4pafhGzUdXqzz6Qz5w1o4IeD7Q+xi
# nrP1mm5ej/WjezjFZKs7sgQ=
# SIG # End signature block
