# Upgrades RadSight UWP to WPF. To be used on Charthouse Provisions systems:
# - RadSight Terminal (Avalue - legacy)
# - RadSight Medical Terminal
# - RadSight Industrial Terminal
# - RadSat/ RadSat2

$versionFilePath = "C:\version.txt"
$toolsPath = "C:\Tools"
$dataSourceFolder = "C:\Users\RadSight\AppData\Local\Packages\RadSight_01d5n8xr87mrw\LocalState\RemoteDNA\"
$dataTargetFolder = "C:\ProgramData\RemoteDNA\"
$application = "RadSight"
$scriptPath = Split-Path -Parent $MyInvocation.MyCommand.Path
$toolZipPath = Join-Path -Path $scriptPath -ChildPath "KioskTools.zip"

function RunSystemChecks {
    $checksFailed = $false

    # ensure that we are running as administrator, otherwise elevate automatically
    if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
        Write-Host "Running as administrator."
        Start-Process powershell -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$PSCommandPath`"" -Verb RunAs
        exit
    }    

    # check we have a tools.zip file in same folder as this script
    if (-not (Test-Path $toolZipPath)) {
        Write-Host "KioskTools.zip file not found in the same folder as this script." -ForegroundColor Red
        $checksFailed = $true
    }    

    # check we are running on a supported system, by seeing if version.txt exists
    if (-not (Test-Path $versionFilePath)) {
        Write-Host "This script must be run on a supported system." -ForegroundColor Red
        $checksFailed = $true
    }

    # check if RadSight UWP installed
    $radsightVersion = (Get-AppPackage $application -AllUsers | Select-Object -ExpandProperty Version)
    if ($null -eq $radsightVersion) {
        Write-Host "RadSight UWP is not installed. Nothing to do." -ForegroundColor Red
        $checksFailed = $true
    }

    # If any checks failed, display all errors and exit
    if ($checksFailed) {
        Read-Host -Prompt "Press Enter to exit"
        exit
    }
}

function UpgradeTools()
{
    # upgrade c:\Tools folder
    Write-Host "Upgrading Tools..." -ForegroundColor Green
    # unzip tools.zip to c:\Tools, and overwrite existing files
    try {
        Expand-Archive -Path $toolZipPath -DestinationPath $toolsPath -Force
        Write-Host "Tools upgraded successfully."
    } catch {
        Write-Host "Failed to upgrade Tools." -ForegroundColor Red
        Read-Host -Prompt "Press Enter to exit"
        exit
    }

    # update disk version (C:\version.txt) to append " (WPF)" to the end of the first line
    Write-Host "Updating version file..."
    $versionFileContent = Get-Content $versionFilePath -Raw
    if (-not $versionFileContent.Contains("WPF")) {
        $lines = $versionFileContent -split "`r`n"
        $lines[0] = $lines[0] + " (WPF)"
        Set-Content -Path $versionFilePath -Value ($lines -join "`r`n")
        Write-Host "Version file updated."
    } else {
        Write-Host "Version file already contains WPF, skipping update."
    }
}

function Copy-DirectoryWithPermissions {
    param (
        [Parameter(Mandatory = $true)]
        [string]$SourcePath,
        
        [Parameter(Mandatory = $true)]
        [string]$DestinationPath
    )

    # Expand any environment variables in the paths
    $expandedSourcePath = [System.Environment]::ExpandEnvironmentVariables($SourcePath)
    $expandedDestPath = [System.Environment]::ExpandEnvironmentVariables($DestinationPath)

    # Test if source path exists
    if (-not (Test-Path -Path $expandedSourcePath)) {
        Write-Error "Source path '$expandedSourcePath' does not exist."
        return $false
    }

    try {
        # Only copy data if destination directory doesn't already exist
        if (-not (Test-Path -Path $expandedDestPath)) {
            New-Item -ItemType Directory -Path $expandedDestPath -Force | Out-Null
            Write-Verbose "Created destination directory: $expandedDestPath"        

            # Copy items with their permissions
            $copyParams = @{
                Path = "$expandedSourcePath\*"
                Destination = $expandedDestPath
                Recurse = $true
                Force = $true
                PassThru = $true
            }

            Copy-Item @copyParams | ForEach-Object {
                # Get and copy the ACL from source to destination
                $sourcePath = $_.FullName.Replace($expandedDestPath, $expandedSourcePath)
                $acl = Get-Acl -Path $sourcePath
                Set-Acl -Path $_.FullName -AclObject $acl
                Write-Verbose "Copied '$($_.Name)' with permissions"
            }

            Write-Verbose "Directory copy completed successfully"
            return $true
        }

        Write-Verbose "Do not need to copy data as it is already in the correct place."
        return $false
    }
    catch {
        Write-Host "Failed to copy data from $SourcePath to $DestinationPath" -ForegroundColor Red
        Read-Host -Prompt "Press Enter to exit"
        exit
        return $false
    }
}

function UpgradeUwpToWpf {
    # disable kiosk mode - run script (C:\tools\Disable-KioskMode.ps1)
    # this will also log the user out - and release any locked files
    $disableKioskModeScriptPath = "$toolsPath\Disable-KioskMode.ps1"
    Write-Host "Disabling kiosk mode..." -ForegroundColor Green
    Start-Process powershell -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$disableKioskModeScriptPath`"" -Verb RunAs -Wait

    # copy UWP self-host data - if exists
    if (Test-Path -Path $dataSourceFolder) { 
        Copy-DirectoryWithPermissions -SourcePath $dataSourceFolder -DestinationPath $dataTargetFolder
    }

    # Capture UWP App Settings and store them in C:\Programdata\RemoteDNA as UWP_exported_settings.json file
    MigrateSettingsUwpToWpf

    # uninstall UWP version of RadSight
    Write-Host "Uninstalling UWP version of RadSight..." -ForegroundColor Green
    $radsightPackage = Get-AppPackage $application -AllUsers
    if ($radsightPackage) {
        $radsightPackage | Remove-AppPackage -AllUsers
        Write-Host "UWP version of RadSight uninstalled successfully."
    } else {
        Write-Host "UWP version of RadSight is not installed." -ForegroundColor Red
    }

    # install WPF version of RadSight - run script (C:\tools\Update-RadSight.ps1)
    Write-Host "Installing WPF version of RadSight..." -ForegroundColor Green
    $updateRadSightScriptPath = "$toolsPath\Update-RadSight.ps1"
    Start-Process powershell -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$updateRadSightScriptPath`"" -Verb RunAs -Wait
    if ($?) {
        Write-Host "WPF version of RadSight installed successfully."
    } else {
        Write-Host "Failed to install WPF version of RadSight." -ForegroundColor Red
        Read-Host -Prompt "Press Enter to exit"
        exit
    }

    # re-enable kiosk mode - run script (C:\tools\Enable-KioskMode.ps1)
    $enableKioskModeScriptPath = "$toolsPath\Enable-KioskMode.ps1"
    Write-Host "Re-enabling kiosk mode..." -ForegroundColor Green
    Start-Process powershell -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$enableKioskModeScriptPath`"" -Verb RunAs -Wait
}

function Get-RawRegistryTypes {
    param(
        [string]$RegistryPath
    )
    
    # Define Windows API functions
    $signature = @'
    [DllImport("advapi32.dll", CharSet = CharSet.Auto)]
    public static extern int RegOpenKeyEx(
        IntPtr hKey,
        string subKey,
        int ulOptions,
        int samDesired,
        out IntPtr hkResult);

    [DllImport("advapi32.dll", CharSet = CharSet.Auto)]
    public static extern int RegEnumValue(
        IntPtr hKey,
        int dwIndex,
        System.Text.StringBuilder lpValueName,
        ref int lpcchValueName,
        IntPtr lpReserved,
        out int lpType,
        IntPtr lpData,
        ref int lpcbData);

    [DllImport("advapi32.dll", CharSet = CharSet.Auto)]
    public static extern int RegQueryValueEx(
        IntPtr hKey,
        string lpValueName,
        IntPtr lpReserved,
        out int lpType,
        byte[] lpData,
        ref int lpcbData);

    [DllImport("advapi32.dll")]
    public static extern int RegCloseKey(IntPtr hKey);

    public static readonly IntPtr HKEY_LOCAL_MACHINE = new IntPtr(-2147483646);
    public static readonly int KEY_READ = 0x20019;
'@

    # Add the Windows API functions to PowerShell with unique type name
    $timestamp = [DateTime]::Now.Ticks
    $typeName = "RegAPI_$timestamp"
    $namespace = "Win32"
    
    $RegAPI = Add-Type -MemberDefinition $signature -Name $typeName -Namespace $namespace -PassThru

    # Convert registry path
    $keyPath = $RegistryPath -replace "HKLM\\", ""
    
    # Open the registry key
    $hKey = [IntPtr]::Zero
    $result = $RegAPI::RegOpenKeyEx($RegAPI::HKEY_LOCAL_MACHINE, $keyPath, 0, $RegAPI::KEY_READ, [ref]$hKey)
    
    if ($result -ne 0) {
        Write-Error "Failed to open registry key. Error code: $result"
        return
    }

    try {
        $results = @()
        $index = 0
        
        while ($true) {
            # Prepare buffers
            $valueName = New-Object System.Text.StringBuilder 256
            $valueNameSize = 256
            $valueType = 0
            $dataSize = 0
            
            # Enumerate the next value
            $enumResult = $RegAPI::RegEnumValue(
                $hKey, 
                $index, 
                $valueName, 
                [ref]$valueNameSize, 
                [IntPtr]::Zero, 
                [ref]$valueType, 
                [IntPtr]::Zero, 
                [ref]$dataSize
            )
            
            if ($enumResult -eq 259) { # ERROR_NO_MORE_ITEMS
                break
            }
            
            if ($enumResult -eq 0) { # ERROR_SUCCESS
                $hexType = "0x{0:x}" -f $valueType
                
                # Get the actual value data using Windows API
                $valueData = Get-RegistryValueData -hKey $hKey -valueName $valueName.ToString() -RegAPI $RegAPI
                
                $results += [PSCustomObject]@{
                    Name = $valueName.ToString()
                    RawType = $valueType
                    HexType = $hexType
                    TypeName = Get-RegistryTypeName -TypeValue $valueType
                    Data = $valueData
                }
            }
            
            $index++
        }
        
        return $results
        
    } finally {
        # Always close the registry key
        $RegAPI::RegCloseKey($hKey) | Out-Null
    }
}

function Get-RegistryValueData {
    param(
        [IntPtr]$hKey,
        [string]$valueName,
        [object]$RegAPI
    )
    
    try {
        # First call to get the data size
        $dataSize = 0
        $dataType = 0
        $result = $RegAPI::RegQueryValueEx($hKey, $valueName, [IntPtr]::Zero, [ref]$dataType, $null, [ref]$dataSize)
        
        if ($result -ne 0 -or $dataSize -eq 0) {
            return "<No data or error reading size>"
        }
        
        # Second call to get the actual data
        $dataBuffer = New-Object byte[] $dataSize
        $result = $RegAPI::RegQueryValueEx($hKey, $valueName, [IntPtr]::Zero, [ref]$dataType, $dataBuffer, [ref]$dataSize)
        
        if ($result -ne 0) {
            return "<Error reading data>"
        }
        
        # Parse data based on type
        switch ($dataType) {
            100000004 { # int
                if ($dataBuffer.Length -ge 4) {
                    return [System.BitConverter]::ToInt32($dataBuffer, 0)
                }
            }
            100000011 { # bool
                if ($dataBuffer.Length -ge 1) {
                    return [bool]($dataBuffer[0] -ne 0)
                }
            }
            100000012 { # string
                # Try different string encodings and termination handling
                if ($dataBuffer.Length -eq 0) {
                    return ""
                }
                
                # First try UTF-16 (Unicode) since this is common for Windows registry
                try {
                    if ($dataBuffer.Length -ge 2) {
                        # Find null terminator for UTF-16 (two consecutive zero bytes)
                        $utf16NullIndex = -1
                        for ($i = 0; $i -lt ($dataBuffer.Length - 1); $i += 2) {
                            if ($dataBuffer[$i] -eq 0 -and $dataBuffer[$i + 1] -eq 0) {
                                $utf16NullIndex = $i
                                break
                            }
                        }
                        
                        if ($utf16NullIndex -ge 0) {
                            $utf16Bytes = $dataBuffer[0..($utf16NullIndex-1)]
                        } else {
                            # No null terminator found, use all data but make sure it's even length
                            $endIndex = if ($dataBuffer.Length % 2 -eq 0) { $dataBuffer.Length - 1 } else { $dataBuffer.Length - 2 }
                            $utf16Bytes = $dataBuffer[0..$endIndex]
                        }
                        
                        if ($utf16Bytes.Length -gt 0) {
                            $utf16String = [System.Text.Encoding]::Unicode.GetString($utf16Bytes)
                            # If it looks like valid text, return it
                            if ($utf16String -and $utf16String.Trim().Length -gt 0) {
                                return $utf16String.Trim()
                            }
                        }
                    }
                } catch { }
                
                # Try UTF-8 with proper null termination
                try {
                    # Find the first null terminator for UTF-8
                    $nullIndex = -1
                    for ($i = 0; $i -lt $dataBuffer.Length; $i++) {
                        if ($dataBuffer[$i] -eq 0) {
                            $nullIndex = $i
                            break
                        }
                    }
                    
                    # If we found a null terminator, use bytes up to that point
                    if ($nullIndex -ge 0 -and $nullIndex -gt 0) {
                        $stringBytes = $dataBuffer[0..($nullIndex-1)]
                    } else {
                        $stringBytes = $dataBuffer
                    }
                    
                    if ($stringBytes.Length -gt 0) {
                        $utf8String = [System.Text.Encoding]::UTF8.GetString($stringBytes)
                        # Check if it contains printable characters
                        if ($utf8String -and $utf8String.Trim().Length -gt 0) {
                            return $utf8String.Trim()
                        }
                    }
                } catch { }
                
                # Fallback: try ASCII
                try {
                    # Remove trailing zeros and try ASCII
                    $cleanBytes = @()
                    foreach ($byte in $dataBuffer) {
                        if ($byte -eq 0) { break }
                        $cleanBytes += $byte
                    }
                    
                    if ($cleanBytes.Count -gt 0) {
                        $asciiString = [System.Text.Encoding]::ASCII.GetString($cleanBytes)
                        return $asciiString.Trim()
                    }
                } catch { }
                
                # Debug: show raw bytes to understand the format
                $hexString = ($dataBuffer | ForEach-Object { $_.ToString("X2") }) -join " "
                return "Debug Hex: $hexString"
            }
            default {
                # For unknown types, show as hex
                $hexString = ($dataBuffer | ForEach-Object { $_.ToString("X2") }) -join " "
                return "Hex: $hexString"
            }
        }
        
        return "<Unknown data format>"
        
    } catch {
        return "<Error: $($_.Exception.Message)>"
    }
}

function Get-RegistryTypeName {
    param([int]$TypeValue)
    
    switch ($TypeValue) {
        0 { return "REG_NONE" }
        1 { return "REG_SZ" }
        2 { return "REG_EXPAND_SZ" }
        3 { return "REG_BINARY" }
        4 { return "REG_DWORD" }
        5 { return "REG_DWORD_BIG_ENDIAN" }
        6 { return "REG_LINK" }
        7 { return "REG_MULTI_SZ" }
        8 { return "REG_RESOURCE_LIST" }
        9 { return "REG_FULL_RESOURCE_DESCRIPTOR" }
        10 { return "REG_RESOURCE_REQUIREMENTS_LIST" }
        11 { return "REG_QWORD" }
        100000004 { return "int" }
        100000011 { return "bool" }
        100000012 { return "string" }
        default { return "CUSTOM_TYPE_$TypeValue" }
    }
}

# Function to migrate settings from UWP to WPF
# This function will extract settings from the UWP app and save them in a JSON file
# in the ProgramData folder for RemoteDNA
function MigrateSettingsUwpToWpf {
    # RadSight local app data path
    $localAppDataPath_RadSight = "C:\Users\RadSight\AppData\Local"

    # Program Data path for RemoteDNA
    $programDataPath = "C:\ProgramData\RemoteDNA"

    # Check if the local app data path exist
    if (-not (Test-Path $localAppDataPath_RadSight)) {
        Write-Host "Local app data path not found at $localAppDataPath_RadSight." -ForegroundColor Red
        exit 1
    }
    # Check if the program data path exist
    if (-not (Test-Path $programDataPath)) {
        Write-Host "Program data path not found at $programDataPath. Creating the folder." 
        New-Item -ItemType Directory -Path $programDataPath -Force | Out-Null
    }

    # Use RadSight's LocalAppData variable for the settings path
    $UwpSettingsPath = $localAppDataPath_RadSight + "\Packages\RadSight_01d5n8xr87mrw\Settings\settings.dat"

    # Check if file exists
    if (-not (Test-Path $UwpSettingsPath)) {
        Write-Host "Error: Settings file not found at $UwpSettingsPath" -ForegroundColor Red
        exit 1
    }

    # Make sure we're running as administrator
    $isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
    if (-not $isAdmin) {
        Write-Host "This script requires administrative privileges. Please run as Administrator." -ForegroundColor Red
        exit 1
    }

    # First unload the hive if it was previously loaded
    reg unload HKLM\_Settings 2>$null

    # Load the settings file as a registry hive
    try {
        reg load HKLM\_Settings "$UwpSettingsPath" | Out-Null
        Write-Host "Successfully loaded settings file as registry hive" -ForegroundColor Green
    } catch {
        Write-Host "Error loading settings file: $_" -ForegroundColor Red
        exit 1
    }

    # Declare $jsonData as an empty hashtable
    $jsonData = @{}
    
    try {
        # Get the HKLM\_Settings\LocalState registry values
        $results = Get-RawRegistryTypes -RegistryPath "HKLM\_Settings\LocalState"
                        
        # Loop through results and add to hashtable
        foreach ($item in $results) {
            $jsonData[$item.Name] = $item.Data
        }               
    } catch {
        Write-Host "Error processing registry data: $_" -ForegroundColor Red
    } finally {
        [System.GC]::Collect()
        Start-Sleep -Seconds 1        
    }

    # Save the extracted Settings as a JSON file
    $exportFilePath = $programDataPath + "\UWP_exported_settings.json"   
    
    $jsonData | ConvertTo-Json -Depth 10 | Out-File -FilePath $exportFilePath -Encoding UTF8 
    Write-Host "UWP RadSight Settings saved to $exportFilePath" -ForegroundColor Green

    # Grant full control permissions to Users group for the exported settings file so RadSight can rename it later
    $acl = Get-Acl $exportFilePath
    $permission = "RadSight","FullControl","Allow"
    $accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule $permission
    $acl.SetAccessRule($accessRule)
    Set-Acl -Path $exportFilePath -AclObject $acl

    # Unload registry hive
    [System.GC]::Collect()
    Start-Sleep -Seconds 1
    # reg unload HKLM\_Settings 2>$null

    Write-Host "`nWPF settings have been successfully migrated from UWP!" -ForegroundColor Green
}

# check system is ready for upgrade, otherwise exit
RunSystemChecks

# upgrade Tools (C:\tools)
UpgradeTools

# Execute the upgrade
UpgradeUwpToWpf

# show countdown for 10seconds
$seconds = 10
for ($i = $seconds; $i -gt 0; $i--) {
    Write-Host "System will reboot in $i seconds. Press Ctrl+C to cancel." -NoNewline
    Start-Sleep -Seconds 1
    Write-Host "`r" -NoNewline
}
Write-Host ""

# reboot
Write-Host "Rebooting system..."
Restart-Computer -Force
# SIG # Begin signature block
# MII+HgYJKoZIhvcNAQcCoII+DzCCPgsCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCC0Se4EtAOAZXQx
# l75L1MpHG0OP4qqMqaXaQgomYHi6saCCIuAwggXMMIIDtKADAgECAhBUmNLR1FsZ
# lUgTecgRwIeZMA0GCSqGSIb3DQEBDAUAMHcxCzAJBgNVBAYTAlVTMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jvc29mdCBJZGVu
# dGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAy
# MDAeFw0yMDA0MTYxODM2MTZaFw00NTA0MTYxODQ0NDBaMHcxCzAJBgNVBAYTAlVT
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xSDBGBgNVBAMTP01pY3Jv
# c29mdCBJZGVudGl0eSBWZXJpZmljYXRpb24gUm9vdCBDZXJ0aWZpY2F0ZSBBdXRo
# b3JpdHkgMjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALORKgeD
# Bmf9np3gx8C3pOZCBH8Ppttf+9Va10Wg+3cL8IDzpm1aTXlT2KCGhFdFIMeiVPvH
# or+Kx24186IVxC9O40qFlkkN/76Z2BT2vCcH7kKbK/ULkgbk/WkTZaiRcvKYhOuD
# PQ7k13ESSCHLDe32R0m3m/nJxxe2hE//uKya13NnSYXjhr03QNAlhtTetcJtYmrV
# qXi8LW9J+eVsFBT9FMfTZRY33stuvF4pjf1imxUs1gXmuYkyM6Nix9fWUmcIxC70
# ViueC4fM7Ke0pqrrBc0ZV6U6CwQnHJFnni1iLS8evtrAIMsEGcoz+4m+mOJyoHI1
# vnnhnINv5G0Xb5DzPQCGdTiO0OBJmrvb0/gwytVXiGhNctO/bX9x2P29Da6SZEi3
# W295JrXNm5UhhNHvDzI9e1eM80UHTHzgXhgONXaLbZ7LNnSrBfjgc10yVpRnlyUK
# xjU9lJfnwUSLgP3B+PR0GeUw9gb7IVc+BhyLaxWGJ0l7gpPKWeh1R+g/OPTHU3mg
# trTiXFHvvV84wRPmeAyVWi7FQFkozA8kwOy6CXcjmTimthzax7ogttc32H83rwjj
# O3HbbnMbfZlysOSGM1l0tRYAe1BtxoYT2v3EOYI9JACaYNq6lMAFUSw0rFCZE4e7
# swWAsk0wAly4JoNdtGNz764jlU9gKL431VulAgMBAAGjVDBSMA4GA1UdDwEB/wQE
# AwIBhjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTIftJqhSobyhmYBAcnz1AQ
# T2ioojAQBgkrBgEEAYI3FQEEAwIBADANBgkqhkiG9w0BAQwFAAOCAgEAr2rd5hnn
# LZRDGU7L6VCVZKUDkQKL4jaAOxWiUsIWGbZqWl10QzD0m/9gdAmxIR6QFm3FJI9c
# Zohj9E/MffISTEAQiwGf2qnIrvKVG8+dBetJPnSgaFvlVixlHIJ+U9pW2UYXeZJF
# xBA2CFIpF8svpvJ+1Gkkih6PsHMNzBxKq7Kq7aeRYwFkIqgyuH4yKLNncy2RtNwx
# AQv3Rwqm8ddK7VZgxCwIo3tAsLx0J1KH1r6I3TeKiW5niB31yV2g/rarOoDXGpc8
# FzYiQR6sTdWD5jw4vU8w6VSp07YEwzJ2YbuwGMUrGLPAgNW3lbBeUU0i/OxYqujY
# lLSlLu2S3ucYfCFX3VVj979tzR/SpncocMfiWzpbCNJbTsgAlrPhgzavhgplXHT2
# 6ux6anSg8Evu75SjrFDyh+3XOjCDyft9V77l4/hByuVkrrOj7FjshZrM77nq81YY
# uVxzmq/FdxeDWds3GhhyVKVB0rYjdaNDmuV3fJZ5t0GNv+zcgKCf0Xd1WF81E+Al
# GmcLfc4l+gcK5GEh2NQc5QfGNpn0ltDGFf5Ozdeui53bFv0ExpK91IjmqaOqu/dk
# ODtfzAzQNb50GQOmxapMomE2gj4d8yu8l13bS3g7LfU772Aj6PXsCyM2la+YZr9T
# 03u4aUoqlmZpxJTG9F9urJh4iIAGXKKy7aIwggcEMIIE7KADAgECAhMzAAYaeTMd
# 9e5ebamiAAAABhp5MA0GCSqGSIb3DQEBDAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMTIk1pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIwHhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0
# MjAzMzAyWjCBgDELMAkGA1UEBhMCR0IxDzANBgNVBAgTBkRvcnNldDEOMAwGA1UE
# BxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0
# ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBEYXRhIE1hbmFnZW1lbnQgTHRkMIIBojAN
# BgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAmcc32FHGLV4FiKzse5eN9PjhviRa
# CblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK6e9cz2ayR0wdaIJSElN50pHTPrWMDQSL
# P7wkC1LovyoEhNzVb40CoJhCTXQTkHqnACMD9NqeDdyaF4r0Nb0u716+W67pv/CA
# CE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZVrqE61MYKeWlQWErF2VSZJaA8X9pJuvH
# 1WxVB+Mr7LIehjb9cxmUfelY1LkKimcVkdpSlZsBCZVANliqCugFrqMLJ1jQRmpd
# 8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp/hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8
# vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgXTHt5262ALFRyo3/NShidyL3k8a4hSFwD
# tWFHtXJv7HKGxbCCZdR8TGMFupw4IeWSBdApLGInTzSMhDxAyTK26McpCGRe3c1z
# BOw753ZIhx5k0LhWztmdCXWZIh9h1UBySzdDAgMBAAGjggIaMIICFjAMBgNVHRMB
# Af8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9BgNVHSUENjA0BgorBgEEAYI3YQEABggr
# BgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQU
# uRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYDVR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQW
# hCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZWaHR0cDovL3d3dy5taWNyb3NvZnQuY29t
# L3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSUQlMjBWZXJpZmllZCUyMENTJTIwQU9D
# JTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUFBwEBBIGYMIGVMGQGCCsGAQUFBzAChlho
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUy
# MElEJTIwVmVyaWZpZWQlMjBDUyUyMEFPQyUyMENBJTIwMDIuY3J0MC0GCCsGAQUF
# BzABhiFodHRwOi8vb25lb2NzcC5taWNyb3NvZnQuY29tL29jc3AwZgYDVR0gBF8w
# XTBRBgwrBgEEAYI3TIN9AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9Eb2NzL1JlcG9zaXRvcnkuaHRtMAgGBmeBDAEEATAN
# BgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKMPRNLuXymFInLM6+tzuw0G5i8qcvMna7j
# ltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYcn19Kvk5bdbRQW50EmDEdH5ybIuE1v5VP
# OjGAg9xvXFk1RdISpST9ghRE+JpdI2gx1uKGUGFCjll6nU/WuFpskg2jV6HQVg7t
# cdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBGDrfJJwLty0ageLObO88+fJdES7ZYsmC1
# Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9yQ+UTijQVYml/vi4oOLROYVk9Auf31WB5
# qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U++cDFVsMH6hvGXrByof7kuE6BQbYk2T0
# 8MaNus4xleApIAYyWr8Prtwe71+T2S+OM/NE+T0JwATBrSwUjBftUzM9nQ3iHGJW
# gT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcbQmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1
# Xjau+kf/EJeYzrhw1H5++1xVbSL1a+BudVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5o
# ERoxq92UpXtQYMo8AOC0TqmdFsqrAmmqi+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJ
# wyGaI2eSqCeROzqTE/BUP6ICU26FD03Mr1QnmMVAN3ligIAAAHHTMJmNaLHVuhVU
# DCAwggcEMIIE7KADAgECAhMzAAYaeTMd9e5ebamiAAAABhp5MA0GCSqGSIb3DQEB
# DAUAMFoxCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xKzApBgNVBAMTIk1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDIw
# HhcNMjUxMTExMjAzMzAyWhcNMjUxMTE0MjAzMzAyWjCBgDELMAkGA1UEBhMCR0Ix
# DzANBgNVBAgTBkRvcnNldDEOMAwGA1UEBxMFUG9vbGUxJzAlBgNVBAoTHkNoYXJ0
# aG91c2UgRGF0YSBNYW5hZ2VtZW50IEx0ZDEnMCUGA1UEAxMeQ2hhcnRob3VzZSBE
# YXRhIE1hbmFnZW1lbnQgTHRkMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC
# AYEAmcc32FHGLV4FiKzse5eN9PjhviRaCblNvBtnqrPnUAYYuJnfTT+uo4/IDSXK
# 6e9cz2ayR0wdaIJSElN50pHTPrWMDQSLP7wkC1LovyoEhNzVb40CoJhCTXQTkHqn
# ACMD9NqeDdyaF4r0Nb0u716+W67pv/CACE8Lpi9bRQQT1aUJJlRG57wzPESwkDyZ
# VrqE61MYKeWlQWErF2VSZJaA8X9pJuvH1WxVB+Mr7LIehjb9cxmUfelY1LkKimcV
# kdpSlZsBCZVANliqCugFrqMLJ1jQRmpd8xgaGT3GwW5Q4iQrKyn4Dw2lDVncgTMp
# /hoyNIQzYfUw44hS0zfo8ZXOUjptAqN8vmAE5ugAdxwmAts0TtK6RwS7xrAm1dgX
# THt5262ALFRyo3/NShidyL3k8a4hSFwDtWFHtXJv7HKGxbCCZdR8TGMFupw4IeWS
# BdApLGInTzSMhDxAyTK26McpCGRe3c1zBOw753ZIhx5k0LhWztmdCXWZIh9h1UBy
# SzdDAgMBAAGjggIaMIICFjAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDA9
# BgNVHSUENjA0BgorBgEEAYI3YQEABggrBgEFBQcDAwYcKwYBBAGCN2GDq+jHTIHw
# 4u1igoD5nDWBpKyqVzAdBgNVHQ4EFgQUuRPRKwj4iXWQUhjHlE/YCMqM9e8wHwYD
# VR0jBBgwFoAUJEWZoXeQKnzDyoOwbmQWhCr4LGcwZwYDVR0fBGAwXjBcoFqgWIZW
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIw
# SUQlMjBWZXJpZmllZCUyMENTJTIwQU9DJTIwQ0ElMjAwMi5jcmwwgaUGCCsGAQUF
# BwEBBIGYMIGVMGQGCCsGAQUFBzAChlhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
# cGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMElEJTIwVmVyaWZpZWQlMjBDUyUyMEFP
# QyUyMENBJTIwMDIuY3J0MC0GCCsGAQUFBzABhiFodHRwOi8vb25lb2NzcC5taWNy
# b3NvZnQuY29tL29jc3AwZgYDVR0gBF8wXTBRBgwrBgEEAYI3TIN9AQEwQTA/Bggr
# BgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9Eb2NzL1Jl
# cG9zaXRvcnkuaHRtMAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAkYGvQxKM
# PRNLuXymFInLM6+tzuw0G5i8qcvMna7jltVc7EtRsV4KoOBZNJ9auRAvdhFrqVYc
# n19Kvk5bdbRQW50EmDEdH5ybIuE1v5VPOjGAg9xvXFk1RdISpST9ghRE+JpdI2gx
# 1uKGUGFCjll6nU/WuFpskg2jV6HQVg7tcdvIDy3hgyZtsnDhmAVL5+qB6vyX/OBG
# DrfJJwLty0ageLObO88+fJdES7ZYsmC1Gn9Vou6rJYRvawWw/H5nIeRcXXwRes9y
# Q+UTijQVYml/vi4oOLROYVk9Auf31WB5qyasiq/VGV8U6QQR/8auaen0Wn1VUm9U
# ++cDFVsMH6hvGXrByof7kuE6BQbYk2T08MaNus4xleApIAYyWr8Prtwe71+T2S+O
# M/NE+T0JwATBrSwUjBftUzM9nQ3iHGJWgT0+1Yh0b/eKcGRbTt9q3M79WJFkvrcb
# QmqMUhG8Xgrfykf6Bp6AvCN5T03yiVq1Xjau+kf/EJeYzrhw1H5++1xVbSL1a+Bu
# dVAQ4HhrKAE5wR6Ca37NFq4tTI4BGR5oERoxq92UpXtQYMo8AOC0TqmdFsqrAmmq
# i+7Hh1sQhS3IpSS9BBlqfRUUZdPs/ATJwyGaI2eSqCeROzqTE/BUP6ICU26FD03M
# r1QnmMVAN3ligIAAAHHTMJmNaLHVuhVUDCAwggdaMIIFQqADAgECAhMzAAAABJZQ
# S9Lb7suIAAAAAAAEMA0GCSqGSIb3DQEBDAUAMGMxCzAJBgNVBAYTAlVTMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xNDAyBgNVBAMTK01pY3Jvc29mdCBJ
# RCBWZXJpZmllZCBDb2RlIFNpZ25pbmcgUENBIDIwMjEwHhcNMjEwNDEzMTczMTUy
# WhcNMjYwNDEzMTczMTUyWjBaMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSswKQYDVQQDEyJNaWNyb3NvZnQgSUQgVmVyaWZpZWQg
# Q1MgQU9DIENBIDAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA4c6g
# 6DOiY6bAOwCPbBlQF2tjo3ckUZuab5ZorMnRp4rOmwZDiTbIpzFkZ/k8k4ivBJV1
# w5/b/oykI+eXAqaaxMdyAO0ModnEW7InfQ+rTkykEzHxRbCNg6KDsTnYc/YdL7II
# iJli8k51upaHLL7CYm9YNc0SFYvlaFj2O0HjO9y/NRmcWNjamZOlRjxW2cWgUsUd
# azSHgRCek87V2bM/17b+o8WXUW91IpggRasmiZ65WEFHXKbyhm2LbhBK6ZWmQoFe
# E+GWrKWCGK/q/4RiTaMNhHXWvWv+//I58UtOxVi3DaK1fQ6YLyIIGHzD4CmtcrGi
# vxupq/crrHunGNB7//Qmul2ZP9HcOmY/aptgUnwT+20g/A37iDfuuVw6yS2Lo0/k
# p/jb+J8vE4FMqIiwxGByL482PMVBC3qd/NbFQa8Mmj6ensU+HEqv9ar+AbcKwumb
# ZqJJKmQrGaSNdWfk2NodgcWOmq7jyhbxwZOjnLj0/bwnsUNcNAe09v+qiozyQQes
# 8A3UXPcRQb8G+c0yaO2ICifWTK7ySuyUJ88k1mtN22CNftbjitiAeafoZ9Vmhn5R
# fb+S/K5arVvTcLukt5PdTDQxl557EIE6A+6XFBpdsjOzkLzdEh7ELk8PVPMjQfPC
# gKtJ84c17fd2C9+pxF1lEQUFXY/YtCL+Nms9cWUCAwEAAaOCAg4wggIKMA4GA1Ud
# DwEB/wQEAwIBhjAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUJEWZoXeQKnzD
# yoOwbmQWhCr4LGcwVAYDVR0gBE0wSzBJBgRVHSAAMEEwPwYIKwYBBQUHAgEWM2h0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5Lmh0
# bTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTASBgNVHRMBAf8ECDAGAQH/AgEA
# MB8GA1UdIwQYMBaAFNlBKbAPD2Ns72nX9c0pnqRIajDmMHAGA1UdHwRpMGcwZaBj
# oGGGX2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY3Jvc29m
# dCUyMElEJTIwVmVyaWZpZWQlMjBDb2RlJTIwU2lnbmluZyUyMFBDQSUyMDIwMjEu
# Y3JsMIGuBggrBgEFBQcBAQSBoTCBnjBtBggrBgEFBQcwAoZhaHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBJRCUyMFZlcmlm
# aWVkJTIwQ29kZSUyMFNpZ25pbmclMjBQQ0ElMjAyMDIxLmNydDAtBggrBgEFBQcw
# AYYhaHR0cDovL29uZW9jc3AubWljcm9zb2Z0LmNvbS9vY3NwMA0GCSqGSIb3DQEB
# DAUAA4ICAQBnLThdlbMNIokdKtzSa8io+pEO95Cc3VOyY/hQsIIcdMyk2hJOzLt/
# M1WXfQyElDk/QtyLzX63TdOb5J+nO8t0pzzwi7ZYvMiNqKvAQO50sMOJn3T3hCPp
# pxNNhoGFVxz2UyiQ4b2vOrcsLK9TOEFXWbUMJObR9PM0wZsABIhu4k6VVLxEDe0G
# SeQX/ZE7PHfTg44Luft4IKqYmnv1Cuosp3glFYsVegLnMWZUZ8UtO9F8QCiAouJY
# hL5OlCksgDb9ve/HQhLFnelfg6dQubIFsqB9IlConYKJZ/HaMZvYtA7y9EORK4cx
# lvTetCXAHayiSXH0ueE/T92wVG0csv5VdUyj6yVrm22vlKYAkXINKvDOB8+s4h+T
# gShlUa2ACu2FWn7JzlTSbpk0IE8REuYmkuyE/BTkk93WDMx7PwLnn4J+5fkvbjjQ
# 08OewfpMhh8SuPdQKqmZ40I4W2UyJKMMTbet16JFimSqDChgnCB6lwlpe0gfbo97
# U7prpbfBKp6B2k2f7Y+TjWrQYN+OdcPOyQAdxGGPBwJSaJG3ohdklCxgAJ5anCxe
# Yl7SjQ5Eua6atjIeVhN0KfPLFPpYz5CQU+JC2H79x4d/O6YOFR9aYe54/CGup7dR
# UIfLSv1/j0DPc6Elf3YyWxloWj8yeY3kHrZFaAlRMwhAXyPQ3rEX9zCCB54wggWG
# oAMCAQICEzMAAAAHh6M0o3uljhwAAAAAAAcwDQYJKoZIhvcNAQEMBQAwdzELMAkG
# A1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjFIMEYGA1UE
# AxM/TWljcm9zb2Z0IElkZW50aXR5IFZlcmlmaWNhdGlvbiBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDIwMB4XDTIxMDQwMTIwMDUyMFoXDTM2MDQwMTIwMTUy
# MFowYzELMAkGA1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjE0MDIGA1UEAxMrTWljcm9zb2Z0IElEIFZlcmlmaWVkIENvZGUgU2lnbmluZyBQ
# Q0EgMjAyMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALLwwK8ZiCji
# 3VR6TElsaQhVCbRS/3pK+MHrJSj3Zxd3KU3rlfL3qrZilYKJNqztA9OQacr1AwoN
# cHbKBLbsQAhBnIB34zxf52bDpIO3NJlfIaTE/xrweLoQ71lzCHkD7A4As1Bs076I
# u+mA6cQzsYYH/Cbl1icwQ6C65rU4V9NQhNUwgrx9rGQ//h890Q8JdjLLw0nV+ayQ
# 2Fbkd242o9kH82RZsH3HEyqjAB5a8+Ae2nPIPc8sZU6ZE7iRrRZywRmrKDp5+Tcm
# JX9MRff241UaOBs4NmHOyke8oU1TYrkxh+YeHgfWo5tTgkoSMoayqoDpHOLJs+qG
# 8Tvh8SnifW2Jj3+ii11TS8/FGngEaNAWrbyfNrC69oKpRQXY9bGH6jn9NEJv9weF
# xhTwyvx9OJLXmRGbAUXN1U9nf4lXezky6Uh/cgjkVd6CGUAf0K+Jw+GE/5VpIVbc
# Nr9rNE50Sbmy/4RTCEGvOq3GhjITbCa4crCzTTHgYYjHs1NbOc6brH+eKpWLtr+b
# Gecy9CrwQyx7S/BfYJ+ozst7+yZtG2wR461uckFu0t+gCwLdN0A6cFtSRtR8bvxV
# FyWwTtgMMFRuBa3vmUOTnfKLsLefRaQcVTgRnzeLzdpt32cdYKp+dhr2ogc+qM6K
# 4CBI5/j4VFyC4QFeUP2YAidLtvpXRRo3AgMBAAGjggI1MIICMTAOBgNVHQ8BAf8E
# BAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNlBKbAPD2Ns72nX9c0p
# nqRIajDmMFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wGQYJ
# KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSME
# GDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYDVR0fBH0wezB5oHegdYZzaHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwSWRl
# bnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIwQ2VydGlmaWNhdGUlMjBBdXRo
# b3JpdHklMjAyMDIwLmNybDCBwwYIKwYBBQUHAQEEgbYwgbMwgYEGCCsGAQUFBzAC
# hnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29m
# dCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIwUm9vdCUyMENlcnRpZmljYXRl
# JTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwLQYIKwYBBQUHMAGGIWh0dHA6Ly9vbmVv
# Y3NwLm1pY3Jvc29mdC5jb20vb2NzcDANBgkqhkiG9w0BAQwFAAOCAgEAfyUqnv7U
# q+rdZgrbVyNMul5skONbhls5fccPlmIbzi+OwVdPQ4H55v7VOInnmezQEeW4LqK0
# wja+fBznANbXLB0KrdMCbHQpbLvG6UA/Xv2pfpVIE1CRFfNF4XKO8XYEa3oW8oVH
# +KZHgIQRIwAbyFKQ9iyj4aOWeAzwk+f9E5StNp5T8FG7/VEURIVWArbAzPt9ThVN
# 3w1fAZkF7+YU9kbq1bCR2YD+MtunSQ1Rft6XG7b4e0ejRA7mB2IoX5hNh3UEauY0
# byxNRG+fT2MCEhQl9g2i2fs6VOG19CNep7SquKaBjhWmirYyANb0RJSLWjinMLXN
# OAga10n8i9jqeprzSMU5ODmrMCJE12xS/NWShg/tuLjAsKP6SzYZ+1Ry358ZTFcx
# 0FS/mx2vSoU8s8HRvy+rnXqyUJ9HBqS0DErVLjQwK8VtsBdekBmdTbQVoCgPCqr+
# PDPB3xajYnzevs7eidBsM71PINK2BoE2UfMwxCCX3mccFgx6UsQeRSdVVVNSyALQ
# e6PT12418xon2iDGE81OGCreLzDcMAZnrUAx4XQLUz6ZTl65yPUiOh3k7Yww94lD
# f+8oG2oZmDh5O1Qe38E+M3vhKwmzIeoB1dVLlz4i3IpaDcR+iuGjH2TdaC1ZOmBX
# iCRKJLj4DT2uhJ04ji+tHD6n58vhavFIrmcxghqUMIIakAIBATBxMFoxCzAJBgNV
# BAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKzApBgNVBAMT
# Ik1pY3Jvc29mdCBJRCBWZXJpZmllZCBDUyBBT0MgQ0EgMDICEzMABhp5Mx317l5t
# qaIAAAAGGnkwDQYJYIZIAWUDBAIBBQCgXjAQBgorBgEEAYI3AgEMMQIwADAZBgkq
# hkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAvBgkqhkiG9w0BCQQxIgQgtxAx3E7SUJR1
# XNC7RWfftPt+vOsGV4wyFVFOB2cA9QYwDQYJKoZIhvcNAQEBBQAEggGAcfMRTy/4
# beRdNApNycP6sQqH+kqqQ2aGM0YXoFHwtBfj3KIglmPwIcEAM4cu1XPkpnm7+biN
# E5GE7Vx8ANsiZLvTuKTFBkhzbHqmcMy3bFLbuevKv/sO2Enwylj+GQEeSxXtsPgw
# vJQYn3Nhrbp2iVRYkEN6xaCf9cuz6Er8cWkpKjkwkNkZ0qrAUWBgATzGSYoMUnec
# KH8D/YXi4cX8PQJ7utBFGW8aSUcmTqk4OPaEJOuKp8ccmdtKr2fF9pnUC9LSL1H3
# dDYCLB8BaCZQ4wBhanTKhDzkNJFrlDsMEMKXOTov3eaKS9z0g0kaGF3HL3h1kCf6
# +NAJX2/M9CBPJvGod/ar6XCAT0DOku1wuceIhI9pGwJxpHPgpcw+Fl9+KcfZWfCU
# fkc07D8tqCdIGiE7ZIt/g0ut4dDW7Fi7bRAQVpFH/97UoWZRJFJu8I/utu3ROEjB
# yz+MzIOMDwNxzSO3nY2FjCkoRAUDo7HIHV9xejb2sse7aNL1YxHI2HJIoYIYFDCC
# GBAGCisGAQQBgjcDAwExghgAMIIX/AYJKoZIhvcNAQcCoIIX7TCCF+kCAQMxDzAN
# BglghkgBZQMEAgEFADCCAWIGCyqGSIb3DQEJEAEEoIIBUQSCAU0wggFJAgEBBgor
# BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIFcpcy096IVIPG3VsCjcBS8Mni+6
# N26zlhJ5t+JUgKX2AgZpBSTt8RsYEzIwMjUxMTEyMTAzMjA4LjY3NVowBIACAfSg
# geGkgd4wgdsxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJTAj
# BgNVBAsTHE1pY3Jvc29mdCBBbWVyaWNhIE9wZXJhdGlvbnMxJzAlBgNVBAsTHm5T
# aGllbGQgVFNTIEVTTjo3QTAwLTA1RTAtRDk0NzE1MDMGA1UEAxMsTWljcm9zb2Z0
# IFB1YmxpYyBSU0EgVGltZSBTdGFtcGluZyBBdXRob3JpdHmggg8hMIIHgjCCBWqg
# AwIBAgITMwAAAAXlzw//Zi7JhwAAAAAABTANBgkqhkiG9w0BAQwFADB3MQswCQYD
# VQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMUgwRgYDVQQD
# Ez9NaWNyb3NvZnQgSWRlbnRpdHkgVmVyaWZpY2F0aW9uIFJvb3QgQ2VydGlmaWNh
# dGUgQXV0aG9yaXR5IDIwMjAwHhcNMjAxMTE5MjAzMjMxWhcNMzUxMTE5MjA0MjMx
# WjBhMQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MTIwMAYDVQQDEylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0Eg
# MjAyMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJ5851Jj/eDFnwV9
# Y7UGIqMcHtfnlzPREwW9ZUZHd5HBXXBvf7KrQ5cMSqFSHGqg2/qJhYqOQxwuEQXG
# 8kB41wsDJP5d0zmLYKAY8Zxv3lYkuLDsfMuIEqvGYOPURAH+Ybl4SJEESnt0MbPE
# oKdNihwM5xGv0rGofJ1qOYSTNcc55EbBT7uq3wx3mXhtVmtcCEr5ZKTkKKE1CxZv
# NPWdGWJUPC6e4uRfWHIhZcgCsJ+sozf5EeH5KrlFnxpjKKTavwfFP6XaGZGWUG8T
# ZaiTogRoAlqcevbiqioUz1Yt4FRK53P6ovnUfANjIgM9JDdJ4e0qiDRm5sOTiEQt
# BLGd9Vhd1MadxoGcHrRCsS5rO9yhv2fjJHrmlQ0EIXmp4DhDBieKUGR+eZ4CNE3c
# tW4uvSDQVeSp9h1SaPV8UWEfyTxgGjOsRpeexIveR1MPTVf7gt8hY64XNPO6iyUG
# sEgt8c2PxF87E+CO7A28TpjNq5eLiiunhKbq0XbjkNoU5JhtYUrlmAbpxRjb9tSr
# eDdtACpm3rkpxp7AQndnI0Shu/fk1/rE3oWsDqMX3jjv40e8KN5YsJBnczyWB4Jy
# eeFMW3JBfdeAKhzohFe8U5w9WuvcP1E8cIxLoKSDzCCBOu0hWdjzKNu8Y5SwB1lt
# 5dQhABYyzR3dxEO/T1K/BVF3rV69AgMBAAGjggIbMIICFzAOBgNVHQ8BAf8EBAMC
# AYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFGtpKDo1L0hjQM972K9J6T7Z
# PdshMFQGA1UdIARNMEswSQYEVR0gADBBMD8GCCsGAQUFBwIBFjNodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL0RvY3MvUmVwb3NpdG9yeS5odG0wEwYDVR0l
# BAwwCgYIKwYBBQUHAwgwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwDwYDVR0T
# AQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTIftJqhSobyhmYBAcnz1AQT2ioojCBhAYD
# VR0fBH0wezB5oHegdYZzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9j
# cmwvTWljcm9zb2Z0JTIwSWRlbnRpdHklMjBWZXJpZmljYXRpb24lMjBSb290JTIw
# Q2VydGlmaWNhdGUlMjBBdXRob3JpdHklMjAyMDIwLmNybDCBlAYIKwYBBQUHAQEE
# gYcwgYQwgYEGCCsGAQUFBzAChnVodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
# b3BzL2NlcnRzL01pY3Jvc29mdCUyMElkZW50aXR5JTIwVmVyaWZpY2F0aW9uJTIw
# Um9vdCUyMENlcnRpZmljYXRlJTIwQXV0aG9yaXR5JTIwMjAyMC5jcnQwDQYJKoZI
# hvcNAQEMBQADggIBAF+Idsd+bbVaFXXnTHho+k7h2ESZJRWluLE0Oa/pO+4ge/XE
# izXvhs0Y7+KVYyb4nHlugBesnFqBGEdC2IWmtKMyS1OWIviwpnK3aL5JedwzbeBF
# 7POyg6IGG/XhhJ3UqWeWTO+Czb1c2NP5zyEh89F72u9UIw+IfvM9lzDmc2O2END7
# MPnrcjWdQnrLn1Ntday7JSyrDvBdmgbNnCKNZPmhzoa8PccOiQljjTW6GePe5sGF
# uRHzdFt8y+bN2neF7Zu8hTO1I64XNGqst8S+w+RUdie8fXC1jKu3m9KGIqF4aldr
# YBamyh3g4nJPj/LR2CBaLyD+2BuGZCVmoNR/dSpRCxlot0i79dKOChmoONqbMI8m
# 04uLaEHAv4qwKHQ1vBzbV/nG89LDKbRSSvijmwJwxRxLLpMQ/u4xXxFfR4f/gksS
# kbJp7oqLwliDm/h+w0aJ/U5ccnYhYb7vPKNMN+SZDWycU5ODIRfyoGl59BsXR/Hp
# RGtiJquOYGmvA/pk5vC1lcnbeMrcWD/26ozePQ/TWfNXKBOmkFpvPE8CH+EeGGWz
# qTCjdAsno2jzTeNSxlx3glDGJgcdz5D/AAxw9Sdgq/+rY7jjgs7X6fqPTXPmaCAJ
# KVHAP19oEjJIBwD1LyHbaEgBxFCogYSOiUIr0Xqcr1nJfiWG2GwYe6ZoAF1bMIIH
# lzCCBX+gAwIBAgITMwAAAFhlzes/odf80gAAAAAAWDANBgkqhkiG9w0BAQwFADBh
# MQswCQYDVQQGEwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIw
# MAYDVQQDEylNaWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAy
# MDAeFw0yNTEwMjMyMDQ2NTVaFw0yNjEwMjIyMDQ2NTVaMIHbMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1l
# cmljYSBPcGVyYXRpb25zMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046N0EwMC0w
# NUUwLUQ5NDcxNTAzBgNVBAMTLE1pY3Jvc29mdCBQdWJsaWMgUlNBIFRpbWUgU3Rh
# bXBpbmcgQXV0aG9yaXR5MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# nXg0pHaQ7PVAlln+HZZrJFcLoKbekhW1yL+QNBUgFFUsjZIKaqqN4oIJsJM3ps0r
# JNSO7ndCNRuZDX2Wgur3Ak77eXrloBXqZmO6ZVXeDNRCLldW4A0/NfjzJ7XXkdEh
# jr81ghXEpR7zC+wbaNN+sPSxzLAZBeibDFP7Xws5wX0ZtIsN1a2+Xq5bvWp3kRMy
# twskTjunRgeLZL/tBp237JVdRPFAQ9jYRKpCqUBo/v1xjBLRCV3PalKjnGfb3MN4
# U7jVyqifFHShcnW5CERRoBmUa6sygDzFSr8e3g93TPNLFUivUE0GmLfbX5ceD1Gt
# 1FcZ6x/JLVATzk5+BWHbMxwJIVkVPTqSSMjQ6KTKdcnq3pH0c4AFJp/glvcpq0U9
# fzZIjJGGvdpishlRl77RQtUhSjxHvCn3LC/xqQQwOHSQDsGh6NX2D0RfsSyEtTAB
# yAae+2w1HByTDTcmlTNLEuQLeCj1gNBdIWj0WOYyDtjjQ/8iTWY6ey1vb9qHljIj
# 5HgIndT5P9MYk2Vg2e7hKUZNBNbA/hsgBsuoZ+IX89WvjEN9abF91S4OJVuinmKs
# LO/MLbnl7ikuD0dN6oA0YewyDQncs12sM9HOtu72QA/TZlefvW8r9xtMXAYoQlcG
# jsk8W4Uc7cfqVqbIPjdoc8ZxBzLcXcVyP4p5cyLwvkMCAwEAAaOCAcswggHHMB0G
# A1UdDgQWBBRyjU3Fer4VxXJ+hjPcRJnxnRIJsDAfBgNVHSMEGDAWgBRraSg6NS9I
# Y0DPe9ivSek+2T3bITBsBgNVHR8EZTBjMGGgX6BdhltodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL2NybC9NaWNyb3NvZnQlMjBQdWJsaWMlMjBSU0ElMjBU
# aW1lc3RhbXBpbmclMjBDQSUyMDIwMjAuY3JsMHkGCCsGAQUFBwEBBG0wazBpBggr
# BgEFBQcwAoZdaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9N
# aWNyb3NvZnQlMjBQdWJsaWMlMjBSU0ElMjBUaW1lc3RhbXBpbmclMjBDQSUyMDIw
# MjAuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYD
# VR0PAQH/BAQDAgeAMGYGA1UdIARfMF0wUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYB
# BQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBv
# c2l0b3J5Lmh0bTAIBgZngQwBBAIwDQYJKoZIhvcNAQEMBQADggIBAHvrxIiVF1iH
# cXvxrJTCD8eOtbPUbK9x+Lz70iYehh+G0UoOMcMf04QD6tQPTeZ5HhGETkcn0raD
# J5NpfbRBuKEH31rxbZK97o12KRDNJ3Nu4ePaUIpH/TcWz8PLVOCECywSxbEgEG20
# kyydGc46c591tXzpfkJDckjoYrypaerdeQLRQH9LaoTYZfdAzMo+Dy0O1DzFJkF5
# YnsmAM8lt9r1NtXdFjdbFMCbV5dau64mV22s186A8Umi+l239+Ue0cbJQIykWhIl
# hhWhxQgoksqHz7kp2GFZAAeySTmIOQOWyXOA8JA8TISJyn3JDOgStv583P3V0QSA
# LT6JXDCW26FV208VGJMzkv0S22iOTZJ/oamTpk8RzD8oWT8pfbe1q/k/bxPiXYRb
# zps96a5YOko7n0Vdo61DOJhL/mhk01Y348gq6vhG/VTcdGHh1rCkwOM05B35AZZq
# 9AtPpfRzJinrHzzGRx+r6fD3ccYMPMMX/Nwd2irzrph172fQcSf1fMwvwIhmfH4G
# WJJ+mf1HA6uXoAOVByckguXvlj8gPi7T2ES6RU8+QssfqTNTJKjsBheWKWv2W4ES
# Ven2L7lCz7i79FhA+0kp0yXJnYwdzWS0ovTINULINmzVyMcSUm5WuVf8YZ33cAud
# 2Opr6N1+RuLZDavDvjiehlI5dH+GEy56MYIHRjCCB0ICAQEweDBhMQswCQYDVQQG
# EwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylN
# aWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMAITMwAAAFhl
# zes/odf80gAAAAAAWDANBglghkgBZQMEAgEFAKCCBJ8wEQYLKoZIhvcNAQkQAg8x
# AgUAMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcN
# MjUxMTEyMTAzMjA4WjAvBgkqhkiG9w0BCQQxIgQgiJvLugqloG3gtMdLKExuxFZx
# ERKNDg+VLQrqvH2a9MswgbkGCyqGSIb3DQEJEAIvMYGpMIGmMIGjMIGgBCDFIlS7
# sgfQ+wAo1cWbWz+WN69VBds58hbran919aLocTB8MGWkYzBhMQswCQYDVQQGEwJV
# UzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNy
# b3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMAITMwAAAFhlzes/
# odf80gAAAAAAWDCCA2EGCyqGSIb3DQEJEAISMYIDUDCCA0yhggNIMIIDRDCCAiwC
# AQEwggEJoYHhpIHeMIHbMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSUwIwYDVQQLExxNaWNyb3NvZnQgQW1lcmljYSBPcGVyYXRpb25zMScwJQYD
# VQQLEx5uU2hpZWxkIFRTUyBFU046N0EwMC0wNUUwLUQ5NDcxNTAzBgNVBAMTLE1p
# Y3Jvc29mdCBQdWJsaWMgUlNBIFRpbWUgU3RhbXBpbmcgQXV0aG9yaXR5oiMKAQEw
# BwYFKw4DAhoDFQCdZHkb26ercF2O62vCdZUfUSvEXKBnMGWkYzBhMQswCQYDVQQG
# EwJVUzEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylN
# aWNyb3NvZnQgUHVibGljIFJTQSBUaW1lc3RhbXBpbmcgQ0EgMjAyMDANBgkqhkiG
# 9w0BAQsFAAIFAOy+zJ4wIhgPMjAyNTExMTIwOTA2MzhaGA8yMDI1MTExMzA5MDYz
# OFowdzA9BgorBgEEAYRZCgQBMS8wLTAKAgUA7L7MngIBADAKAgEAAgIcSwIB/zAH
# AgEAAgISSzAKAgUA7MAeHgIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZ
# CgMCoAowCAIBAAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBCwUAA4IBAQBg
# xTXX97LQbAqA2EbhQFJOxOdOm2D7jBZWi2oNPzrIcqhWcHmh6QcqOwoMc8HXaOWv
# rOBK+LfgotpFWAP6Y58Y3ceXpnoVoXVZN7KocVMyaNFj1NJkAI6V/XHV4JorTTqW
# sTGNc5R69f7mZx4VIQs6DPnRZ9DEOp/oPK7/HFviFpHnJGbOL9wT3wvUAaWji++O
# 12MhH2b8dlpCh+XpkQdm0Oz2qIgGk75hq1+qXSrFAZRF37kYrqBqASMn4rvcOHAK
# D7LfuzXi9oVxJXft5ZBMWEintZNJ5ZGrFEzfc+942NEXL/H8XduRGvIJ9r0oy5mW
# 3P65ES3lpnwVwCx/2fFzMA0GCSqGSIb3DQEBAQUABIICAEH9KNEYG82nogFznOkX
# D/NreDyAWhcAWgU8gi1XjuOWRW4lrW/9J6PXrJ7z7VakFOvomN9cGXno8QS9Jx3n
# WLrl5zforQZ490DTHbzYH5ZCbyt8x9oPyrlm/yS8GA+L8mc6y1ep1jpB0NXTh523
# bdaQG/3iNUsJA0pGbTgxBks4u2csXQQ+ZN9ENQdDbnSS4qtrncxEmOJwvmyVGWg8
# 4TVbEwhsmf+sf2xz6lQlV2X6HYq74xkvnP7NhYNLSTbH27bZhf48vf0D0T4mf7Ze
# tRNlt51raF/WqfUcod3KIA+znJcZD140vfkLqytPQavXCk76xsSX3x0ge4uTmxEo
# 2WDn9SGDymCEcmYDNgg9bx525mRC+XE6hxZDtk6EntD2I+Fhw0NNkf0KCgwPubRV
# y02n4tY7LdB7Sb0ZvRrLRQaJwVZOi1c7cWMvc22p1V1fUgDHaW7x7ZoOZmQKpED/
# yqnTseumPr+e5RocDBavGXDG3/fRBOBWtH1Lk1h098IGnW06j8loGk94tjwn88m5
# HVxTt2Hgr1kGl3bByQx7DUjCFxgMdU9TZk+aJZT6OqI6maJfQ7+8/+/poI8GxY4H
# 3ryjUqZw/YYRqgPxDD2CE5x41/6UObQMDnF5ltE7dIua6IKDRChEUGZwYM9VQvRH
# NPvZvpEtk29ZKMuMmS/2Cv2n
# SIG # End signature block
